"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.addPresetDependencies = addPresetDependencies;
exports.generatePreset = generatePreset;
const devkit_1 = require("@nx/devkit");
const presets_1 = require("../utils/presets");
const versions_1 = require("../../utils/versions");
const get_npm_package_version_1 = require("../utils/get-npm-package-version");
const path_1 = require("path");
const yargsParser = require("yargs-parser");
const child_process_1 = require("child_process");
function addPresetDependencies(host, options) {
    const { dependencies, dev } = getPresetDependencies(options);
    return (0, devkit_1.addDependenciesToPackageJson)(host, dependencies, dev, (0, path_1.join)(options.directory, 'package.json'));
}
function generatePreset(host, opts) {
    const parsedArgs = yargsParser(process.argv, {
        boolean: ['interactive'],
        default: {
            interactive: true,
        },
    });
    const spawnOptions = {
        stdio: 'inherit',
        shell: true,
        cwd: (0, path_1.join)(host.root, opts.directory),
        windowsHide: true,
    };
    const pmc = (0, devkit_1.getPackageManagerCommand)();
    const executable = `${pmc.exec} nx`;
    const args = getPresetArgs(opts);
    return new Promise((resolve, reject) => {
        (0, child_process_1.spawn)(executable, args, spawnOptions).on('close', (code) => {
            if (code === 0) {
                resolve();
            }
            else {
                const message = 'Workspace creation failed, see above.';
                reject(new Error(message));
            }
        });
    });
    function getPresetArgs(options) {
        // supported presets
        return getDefaultArgs(options);
    }
    function getDefaultArgs(opts) {
        return [
            `g`,
            `@nx/workspace:preset`,
            `--name=${opts.appName}`,
            opts.style ? `--style=${opts.style}` : null,
            opts.linter ? `--linter=${opts.linter}` : null,
            opts.preset ? `--preset=${opts.preset}` : null,
            opts.bundler ? `--bundler=${opts.bundler}` : null,
            opts.framework ? `--framework=${opts.framework}` : null,
            opts.docker ? `--docker=${opts.docker}` : null,
            opts.js ? `--js` : null,
            opts.nextAppDir ? '--nextAppDir=true' : '--nextAppDir=false',
            opts.nextSrcDir ? '--nextSrcDir=true' : '--nextSrcDir=false',
            opts.packageManager ? `--packageManager=${opts.packageManager}` : null,
            opts.standaloneApi !== undefined
                ? `--standaloneApi=${opts.standaloneApi}`
                : null,
            parsedArgs.interactive ? '--interactive=true' : '--interactive=false',
            opts.routing !== undefined ? `--routing=${opts.routing}` : null,
            opts.e2eTestRunner !== undefined
                ? `--e2eTestRunner=${opts.e2eTestRunner}`
                : null,
            opts.ssr ? `--ssr` : null,
            opts.prefix !== undefined ? `--prefix=${opts.prefix}` : null,
            opts.nxCloudToken ? `--nxCloudToken=${opts.nxCloudToken}` : null,
        ].filter((e) => !!e);
    }
}
function getPresetDependencies({ preset, presetVersion, bundler, e2eTestRunner, }) {
    switch (preset) {
        case presets_1.Preset.Apps:
        case presets_1.Preset.NPM:
        case presets_1.Preset.TS:
        case presets_1.Preset.TsStandalone:
            return { dependencies: {}, dev: { '@nx/js': versions_1.nxVersion } };
        case presets_1.Preset.AngularMonorepo:
        case presets_1.Preset.AngularStandalone:
            return {
                dependencies: {},
                dev: {
                    '@angular-devkit/core': versions_1.angularCliVersion,
                    '@nx/angular': versions_1.nxVersion,
                    typescript: versions_1.typescriptVersion,
                },
            };
        case presets_1.Preset.Express:
            return { dependencies: {}, dev: { '@nx/express': versions_1.nxVersion } };
        case presets_1.Preset.Nest:
            return {
                dependencies: {},
                dev: { '@nx/nest': versions_1.nxVersion, typescript: versions_1.typescriptVersion },
            };
        case presets_1.Preset.NextJs:
        case presets_1.Preset.NextJsStandalone:
            return { dependencies: { '@nx/next': versions_1.nxVersion }, dev: {} };
        case presets_1.Preset.RemixStandalone:
        case presets_1.Preset.RemixMonorepo:
            return { dependencies: { '@nx/remix': versions_1.nxVersion }, dev: {} };
        case presets_1.Preset.VueMonorepo:
        case presets_1.Preset.VueStandalone:
            return {
                dependencies: {},
                dev: {
                    '@nx/vue': versions_1.nxVersion,
                    '@nx/cypress': e2eTestRunner === 'cypress' ? versions_1.nxVersion : undefined,
                    '@nx/playwright': e2eTestRunner === 'playwright' ? versions_1.nxVersion : undefined,
                    '@nx/vite': versions_1.nxVersion,
                },
            };
        case presets_1.Preset.Nuxt:
        case presets_1.Preset.NuxtStandalone:
            return {
                dependencies: {},
                dev: {
                    '@nx/nuxt': versions_1.nxVersion,
                    '@nx/cypress': e2eTestRunner === 'cypress' ? versions_1.nxVersion : undefined,
                    '@nx/playwright': e2eTestRunner === 'playwright' ? versions_1.nxVersion : undefined,
                },
            };
        case presets_1.Preset.ReactMonorepo:
        case presets_1.Preset.ReactStandalone:
            return {
                dependencies: {},
                dev: {
                    '@nx/react': versions_1.nxVersion,
                    '@nx/cypress': e2eTestRunner !== 'none' ? versions_1.nxVersion : undefined,
                    '@nx/jest': bundler !== 'vite' ? versions_1.nxVersion : undefined,
                    '@nx/vite': bundler === 'vite' ? versions_1.nxVersion : undefined,
                    '@nx/webpack': bundler === 'webpack' ? versions_1.nxVersion : undefined,
                },
            };
        case presets_1.Preset.ReactNative:
            return { dependencies: {}, dev: { '@nx/react-native': versions_1.nxVersion } };
        case presets_1.Preset.Expo:
            return { dependencies: {}, dev: { '@nx/expo': versions_1.nxVersion } };
        case presets_1.Preset.WebComponents:
            return {
                dependencies: {},
                dev: { '@nx/web': versions_1.nxVersion, typescript: versions_1.typescriptVersion },
            };
        case presets_1.Preset.NodeStandalone:
        case presets_1.Preset.NodeMonorepo:
            return {
                dependencies: {},
                dev: {
                    '@nx/node': versions_1.nxVersion,
                    '@nx/webpack': bundler === 'webpack' ? versions_1.nxVersion : undefined,
                },
            };
        default: {
            return {
                dev: {},
                dependencies: {
                    [preset]: process.env['NX_E2E_PRESET_VERSION'] ??
                        (0, get_npm_package_version_1.getNpmPackageVersion)(preset, presetVersion),
                },
            };
        }
    }
}
