"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.updateJestConfig = updateJestConfig;
const devkit_1 = require("@nx/devkit");
const get_source_nodes_1 = require("../../../utilities/typescript/get-source-nodes");
const path_1 = require("path");
const typescript_1 = require("../../../utilities/typescript");
const jest_config_1 = require("../../utils/jest-config");
let tsModule;
function isUsingUtilityFunction(host) {
    const rootConfigPath = (0, jest_config_1.findRootJestConfig)(host);
    if (!rootConfigPath) {
        return false;
    }
    const rootConfig = host.read(rootConfigPath, 'utf-8');
    return (rootConfig.includes('getJestProjects()') ||
        rootConfig.includes('getJestProjectsAsync()'));
}
/**
 * in a standalone project, the root jest.config.ts is a project config instead
 * of multi-project config.
 * in that case we do not need to edit it to remove it
 **/
function isMonorepoConfig(tree) {
    const rootConfigPath = (0, jest_config_1.findRootJestConfig)(tree);
    if (!rootConfigPath) {
        return false;
    }
    return tree.read(rootConfigPath, 'utf-8').includes('projects:');
}
/**
 * Updates the root jest config projects array and removes the project.
 */
function updateJestConfig(tree, schema, projectConfig) {
    if (!tsModule) {
        tsModule = (0, typescript_1.ensureTypescript)();
    }
    const { createSourceFile, ScriptTarget, isPropertyAssignment, isArrayLiteralExpression, isStringLiteral, } = tsModule;
    const projectToRemove = schema.projectName;
    const rootConfigPath = (0, jest_config_1.findRootJestConfig)(tree);
    if (!rootConfigPath ||
        !tree.exists(rootConfigPath) ||
        !tree.exists((0, path_1.join)(projectConfig.root, 'jest.config.ts')) ||
        isUsingUtilityFunction(tree) ||
        !isMonorepoConfig(tree)) {
        return;
    }
    const contents = tree.read(rootConfigPath, 'utf-8');
    const sourceFile = createSourceFile(rootConfigPath, contents, ScriptTarget.Latest);
    const sourceNodes = (0, get_source_nodes_1.getSourceNodes)(sourceFile);
    const projectsAssignment = sourceNodes.find((node) => isPropertyAssignment(node) &&
        node.name.getText(sourceFile) === 'projects' &&
        isArrayLiteralExpression(node.initializer));
    if (!projectsAssignment) {
        throw Error(`Could not remove ${projectToRemove} from projects in /jest.config.ts. Please remove ${projectToRemove} from your projects.`);
    }
    const projectsArray = projectsAssignment.initializer;
    const project = projectsArray.elements.find((item) => isStringLiteral(item) &&
        item.text.startsWith(`<rootDir>/${projectConfig.root}`));
    if (!project) {
        console.warn(`Could not find ${projectToRemove} in projects in /jest.config.ts.`);
        return;
    }
    const previousProject = projectsArray.elements[projectsArray.elements.indexOf(project) - 1];
    const start = previousProject
        ? previousProject.getEnd()
        : project.getStart(sourceFile);
    tree.write(rootConfigPath, (0, devkit_1.applyChangesToString)(contents, [
        {
            type: devkit_1.ChangeType.Delete,
            start,
            length: project.getEnd() - start,
        },
    ]));
}
