"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.default = default_1;
const devkit_1 = require("@nx/devkit");
// nx-ignore-next-line
const path = require("path");
const move_1 = require("../../generators/move/move");
const versions_1 = require("../../utils/versions");
const path_1 = require("path");
const get_import_path_1 = require("../../utilities/get-import-path");
const PROJECT_NAME = 'workspace-plugin';
const DESTINATION = `tools/${PROJECT_NAME}`;
async function default_1(tree) {
    if (!tree.exists('tools/generators')) {
        return;
    }
    const tasks = [];
    if (hasWorkspaceGenerators(tree)) {
        tasks.push(...(await moveWorkspaceGeneratorsToLocalPlugin(tree)));
    }
    removeToolsGeneratorsIfEmpty(tree);
    await (0, devkit_1.formatFiles)(tree);
    return () => {
        for (const task of tasks) {
            task();
        }
    };
}
async function moveWorkspaceGeneratorsToLocalPlugin(tree) {
    const tasks = [];
    let project = (0, devkit_1.getProjects)(tree).get(PROJECT_NAME);
    if (!project) {
        await createNewPlugin(tree);
        tasks.push((0, devkit_1.addDependenciesToPackageJson)(tree, {}, {
            '@nx/plugin': versions_1.nxVersion,
        }));
        project = (0, devkit_1.readProjectConfiguration)(tree, PROJECT_NAME);
    }
    updateExistingPlugin(tree, project);
    return tasks;
}
function hasWorkspaceGenerators(tree) {
    const children = tree.children('tools/generators');
    return (children.length > 0 &&
        !(children.length === 1 && children[0] === '.gitkeep'));
}
function removeToolsGeneratorsIfEmpty(tree) {
    if (!hasWorkspaceGenerators(tree)) {
        tree.delete('tools/generators');
    }
}
// Inspired by packages/nx/src/command-line/workspace-generators.ts
function collectAndMoveGenerators(tree, destinationProjectRoot) {
    const generators = {};
    const generatorsDir = 'tools/generators';
    const destinationDir = (0, devkit_1.joinPathFragments)(destinationProjectRoot, 'src', 'generators');
    for (const c of tree.children('tools/generators')) {
        const childDir = path.join(generatorsDir, c);
        const schemaPath = (0, devkit_1.joinPathFragments)(childDir, 'schema.json');
        if (tree.exists(schemaPath)) {
            const schema = (0, devkit_1.readJson)(tree, schemaPath);
            generators[c] = {
                implementation: `./src/generators/${c}`,
                schema: `./src/generators/${(0, devkit_1.joinPathFragments)(c, 'schema.json')}`,
                description: schema.description ?? `Generator ${c}`,
            };
            moveFilesInDirectory(tree, childDir, (0, devkit_1.joinPathFragments)(destinationDir, c));
        }
    }
    return generators;
}
function moveFilesInDirectory(tree, source, destination) {
    const relative = path_1.posix.relative(source, path_1.posix.dirname(destination));
    if (!relative.startsWith('../')) {
        // If the destination is in the same directory or a subdirectory of the source
        // we can just move the files. If it is not, we need to update the relative imports.
        return;
    }
    let offsetLevel = 0;
    const pathParts = relative.split('/');
    for (const part of pathParts) {
        if (part === '..') {
            offsetLevel++;
        }
        else {
            break;
        }
    }
    for (const c of tree.children(source)) {
        if (!tree.isFile(c)) {
            moveFilesInDirectory(tree, (0, devkit_1.joinPathFragments)(source, c), (0, devkit_1.joinPathFragments)(destination, c));
        }
        tree.rename((0, devkit_1.joinPathFragments)(source, c), (0, devkit_1.joinPathFragments)(destination, c));
        // If its a TS file we can update relative imports with find + replace
        // This could be done with AST, but since we are only looking at relative
        // imports its easy to do via string replace. We replace any strings starting
        // with a relative path outside of their own directory.
        if (c.endsWith('.ts')) {
            let content = tree.read((0, devkit_1.joinPathFragments)(destination, c)).toString();
            // +2 is a bit of a magic number here - represents extra directory levels in a normal
            // plugin structure compared to the workspace-generator layout
            const extraDirectoriesInPluginStructure = 2;
            content = content.replace(new RegExp(`'` + `\.\.\/`.repeat(offsetLevel), 'g'), "'" + '../'.repeat(offsetLevel + extraDirectoriesInPluginStructure));
            content = content.replace(new RegExp(`"` + `\.\.\/`.repeat(offsetLevel), 'g'), '"' + '../'.repeat(offsetLevel + extraDirectoriesInPluginStructure));
            // We write it back in the same spot, since it is moved as if it was a regular file after this
            tree.write((0, devkit_1.joinPathFragments)(source, c), content);
        }
        tree.rename((0, devkit_1.joinPathFragments)(source, c), (0, devkit_1.joinPathFragments)(destination, c));
    }
}
async function createNewPlugin(tree) {
    (0, devkit_1.ensurePackage)('@nx/plugin', versions_1.nxVersion);
    const { pluginGenerator } = 
    // nx-ignore-next-line
    require('@nx/plugin/src/generators/plugin/plugin');
    // nx-ignore-next-line
    const { Linter } = (0, devkit_1.ensurePackage)('@nx/eslint', versions_1.nxVersion);
    const npmScope = (0, get_import_path_1.getNpmScope)(tree);
    const importPath = npmScope ? `@${npmScope}/${PROJECT_NAME}` : PROJECT_NAME;
    await pluginGenerator(tree, {
        minimal: true,
        name: PROJECT_NAME,
        importPath: importPath,
        skipTsConfig: false,
        compiler: 'tsc',
        linter: Linter.EsLint,
        skipFormat: true,
        skipLintChecks: false,
        unitTestRunner: 'jest',
        e2eTestRunner: 'none',
        publishable: false,
    });
    await getCreateGeneratorsJson()(tree, (0, devkit_1.readProjectConfiguration)(tree, PROJECT_NAME).root, PROJECT_NAME);
    await moveGeneratedPlugin(tree, DESTINATION, importPath);
}
function moveGeneratedPlugin(tree, destination, importPath) {
    const config = (0, devkit_1.readProjectConfiguration)(tree, PROJECT_NAME);
    if (config.root !== DESTINATION) {
        return (0, move_1.moveGenerator)(tree, {
            destination,
            projectName: PROJECT_NAME,
            newProjectName: PROJECT_NAME,
            updateImportPath: true,
            importPath: importPath,
            projectNameAndRootFormat: 'as-provided',
        });
    }
}
function updateExistingPlugin(tree, project) {
    const packageJson = (0, devkit_1.readJson)(tree, (0, devkit_1.joinPathFragments)(project.root, 'package.json'));
    const defaultGeneratorsPath = (0, devkit_1.joinPathFragments)(project.root, 'generators.json');
    let generatorsJsonPath = packageJson.generators ||
        packageJson.schematics ||
        tree.exists(defaultGeneratorsPath)
        ? defaultGeneratorsPath
        : null;
    if (!generatorsJsonPath) {
        getCreateGeneratorsJson()(tree, (0, devkit_1.readProjectConfiguration)(tree, PROJECT_NAME).root, PROJECT_NAME);
        generatorsJsonPath = defaultGeneratorsPath;
    }
    (0, devkit_1.updateJson)(tree, generatorsJsonPath, (json) => {
        const generators = collectAndMoveGenerators(tree, project.root);
        json.generators ??= {};
        for (const generator in generators) {
            if (json.generators[generator]) {
                devkit_1.output.warn({
                    title: `Generator ${generator} already exists in ${project.name}`,
                    bodyLines: [
                        'Since you have a generator with the same name in your plugin, the generator from workspace-generators has been discarded.',
                    ],
                });
            }
            else {
                json.generators[generator] = generators[generator];
            }
        }
        return json;
    });
}
function getCreateGeneratorsJson() {
    // We cant use  `as typeof import('@nx/plugin/src/generators/generator/generator');` here
    // because it will cause a typescript error at build time.
    const { createGeneratorsJson } = 
    // nx-ignore-next-line
    require('@nx/plugin/src/generators/generator/generator');
    return createGeneratorsJson;
}
