import { Component, Input, ViewEncapsulation, ChangeDetectionStrategy, ContentChild, Output, EventEmitter } from '@angular/core';
import { sankey, sankeyLeft, sankeyLinkHorizontal } from 'd3-sankey';
import { BaseChartComponent } from '../common/base-chart.component';
import { calculateViewDimensions } from '../common/view-dimensions.helper';
import { ColorHelper } from '../common/color.helper';
import { ScaleType } from '../common/types/scale-type.enum';
import { StyleTypes } from '../common/tooltip/style.type';
import { escapeLabel } from '../common/label.helper';
import { id } from '../utils/id';
import { TextAnchor } from '../common/types/text-anchor.enum';
import * as i0 from "@angular/core";
import * as i1 from "../common/charts/chart.component";
import * as i2 from "@angular/common";
import * as i3 from "../common/tooltip/tooltip.directive";
export class SankeyComponent extends BaseChartComponent {
    constructor() {
        super(...arguments);
        this.showLabels = true;
        this.tooltipDisabled = false;
        this.activeEntries = [];
        this.activate = new EventEmitter();
        this.deactivate = new EventEmitter();
        this.margin = [10, 10, 10, 10];
        this.scaleType = ScaleType.Ordinal;
        this.styleTypes = StyleTypes;
    }
    update() {
        super.update();
        this.dims = calculateViewDimensions({
            width: this.width,
            height: this.height,
            margins: this.margin,
            legendType: this.scaleType
        });
        const linkDefs = this.results;
        const nodeDefs = Array.from(new Set(linkDefs.flatMap(l => [l.source, l.target])), (name) => ({
            name,
            value: linkDefs.filter(l => l.source === name).reduce((acc, l) => acc + l.value, 0)
        }));
        // Configure generator
        const sankeyGenerator = sankey()
            .nodeId(d => d.name)
            .nodeAlign(sankeyLeft)
            .nodeWidth(15)
            .nodePadding(10)
            .extent([
            [1, 5],
            [this.dims.width - 1, this.dims.height - 5]
        ]);
        // Generate links and nodes
        const data = sankeyGenerator({
            nodes: nodeDefs.map(d => Object.assign({}, d)),
            links: linkDefs.map(d => Object.assign({}, d))
        });
        this.valueDomain = this.getValueDomain(data.nodes);
        this.setColors();
        this.nodeRects = data.nodes.map(node => {
            const rect = {
                x: node.x0,
                y: node.y0,
                height: node.y1 - node.y0,
                width: node.x1 - node.x0,
                fill: this.colors.getColor(node.name),
                tooltip: this.getNodeTooltipText(node),
                rx: 5,
                data: {
                    name: node.name,
                    value: node.value
                },
                transform: '',
                label: this.labelFormatting ? this.labelFormatting(node.name) : node.name,
                labelAnchor: TextAnchor.Start
            };
            rect.labelAnchor = this.getTextAnchor(node);
            rect.transform = `translate(${rect.x},${rect.y})`;
            return rect;
        });
        this.linkPaths = data.links.map(link => {
            const gradientId = 'mask' + id().toString();
            const linkPath = {
                path: sankeyLinkHorizontal()(link),
                strokeWidth: Math.max(1, link.width),
                tooltip: this.getLinkTooltipText(link.source, link.target, link.value),
                id: gradientId,
                gradientFill: `url(#${gradientId})`,
                source: link.source,
                target: link.target,
                startColor: this.colors.getColor(link.source.name),
                endColor: this.colors.getColor(link.target.name),
                data: {
                    source: link.source.name,
                    target: link.target.name,
                    value: link.value
                }
            };
            return linkPath;
        });
        this.transform = `translate(${this.dims.xOffset} , ${this.margin[0]})`;
    }
    getNodeTooltipText(node) {
        return `
      <span class="tooltip-label">${escapeLabel(node.name)}</span>
      <span class="tooltip-val">${node.value.toLocaleString()}</span>
    `;
    }
    getLinkTooltipText(sourceNode, targetNode, value) {
        return `
      <span class="tooltip-label">${escapeLabel(sourceNode.name)} • ${escapeLabel(targetNode.name)}</span>
      <span class="tooltip-val">${value.toLocaleString()} (${(value / sourceNode.value).toLocaleString(undefined, {
            style: 'percent',
            maximumFractionDigits: 2
        })})</span>
    `;
    }
    getTextAnchor(node) {
        if (node.layer === 0) {
            return TextAnchor.Start;
        }
        else {
            return TextAnchor.End;
        }
    }
    onClick(data) {
        this.select.emit(data);
    }
    setColors() {
        this.colors = new ColorHelper(this.scheme, this.scaleType, this.valueDomain);
    }
    getValueDomain(nodes) {
        return nodes.map(n => n.name);
    }
}
SankeyComponent.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "13.1.0", ngImport: i0, type: SankeyComponent, deps: null, target: i0.ɵɵFactoryTarget.Component });
SankeyComponent.ɵcmp = i0.ɵɵngDeclareComponent({ minVersion: "12.0.0", version: "13.1.0", type: SankeyComponent, selector: "ngx-charts-sankey", inputs: { showLabels: "showLabels", gradient: "gradient", tooltipDisabled: "tooltipDisabled", activeEntries: "activeEntries", labelFormatting: "labelFormatting" }, outputs: { activate: "activate", deactivate: "deactivate" }, queries: [{ propertyName: "tooltipTemplate", first: true, predicate: ["tooltipTemplate"], descendants: true }], usesInheritance: true, ngImport: i0, template: `
    <ngx-charts-chart [view]="[width, height]" [animations]="animations">
      <svg:g [attr.transform]="transform" class="sankey chart">
        <svg:g
          *ngFor="let link of linkPaths"
          class="link"
          ngx-tooltip
          [tooltipDisabled]="tooltipDisabled"
          [tooltipType]="styleTypes.tooltip"
          [tooltipPlacement]="'top'"
          [tooltipTitle]="tooltipTemplate ? undefined : link.tooltip"
          [tooltipTemplate]="tooltipTemplate"
          [tooltipContext]="link.data"
        >
          <svg:defs>
            <svg:linearGradient
              [attr.id]="link.id"
              gradientUnits="userSpaceOnUse"
              [attr.x1]="link.source.x1"
              [attr.x2]="link.target.x0"
            >
              <svg:stop offset="0%" [attr.stop-color]="link.startColor"></svg:stop>
              <svg:stop offset="100%" [attr.stop-color]="link.endColor"></svg:stop>
            </svg:linearGradient>
          </svg:defs>
          <svg:path
            [attr.d]="link.path"
            [attr.stroke]="link.gradientFill"
            [attr.stroke-width]="link.strokeWidth"
            stroke-opacity="0.5"
            fill="none"
            (click)="select.emit(link.data)"
            (mouseenter)="activate.emit(link.data)"
          ></svg:path>
        </svg:g>

        <svg:g *ngFor="let rect of nodeRects" [attr.transform]="rect.transform" class="node">
          <svg:rect
            [attr.x]="0"
            [attr.y]="0"
            [attr.width]="rect.width"
            [attr.height]="rect.height"
            [attr.fill]="rect.fill"
            ngx-tooltip
            [tooltipDisabled]="tooltipDisabled"
            [tooltipType]="styleTypes.tooltip"
            [tooltipPlacement]="'top'"
            [tooltipTitle]="tooltipTemplate ? undefined : rect.tooltip"
            [tooltipTemplate]="tooltipTemplate"
            [tooltipContext]="rect.data"
            (click)="select.emit(rect.data)"
            (mouseenter)="activate.emit(rect.data)"
          ></svg:rect>
        </svg:g>

        <svg:g *ngFor="let rect of nodeRects" [attr.transform]="rect.transform">
          <svg:text
            *ngIf="showLabels && rect.height > 15"
            class="label"
            [attr.x]="rect.width + 5"
            [attr.y]="rect.height / 2"
            [attr.text-anchor]="rect.labelAnchor"
            dy="0.35em"
            [attr.dx]="rect.labelAnchor === 'end' ? -25 : 0"
          >
            {{ rect.label }}
          </svg:text>
        </svg:g>
      </svg:g>
    </ngx-charts-chart>
  `, isInline: true, styles: [".ngx-charts-outer{animation:chartFadeIn linear .6s}@keyframes chartFadeIn{0%{opacity:0}20%{opacity:0}to{opacity:1}}.ngx-charts{float:left;overflow:visible}.ngx-charts .circle,.ngx-charts .cell,.ngx-charts .bar,.ngx-charts .node,.ngx-charts .link,.ngx-charts .arc{cursor:pointer}.ngx-charts .bar.active,.ngx-charts .bar:hover,.ngx-charts .cell.active,.ngx-charts .cell:hover,.ngx-charts .arc.active,.ngx-charts .arc:hover,.ngx-charts .node.active,.ngx-charts .node:hover,.ngx-charts .link.active,.ngx-charts .link:hover,.ngx-charts .card.active,.ngx-charts .card:hover{opacity:.8;transition:opacity .1s ease-in-out}.ngx-charts .bar:focus,.ngx-charts .cell:focus,.ngx-charts .arc:focus,.ngx-charts .node:focus,.ngx-charts .link:focus,.ngx-charts .card:focus{outline:none}.ngx-charts .bar.hidden,.ngx-charts .cell.hidden,.ngx-charts .arc.hidden,.ngx-charts .node.hidden,.ngx-charts .link.hidden,.ngx-charts .card.hidden{display:none}.ngx-charts g:focus{outline:none}.ngx-charts .line-series.inactive,.ngx-charts .line-series-range.inactive,.ngx-charts .polar-series-path.inactive,.ngx-charts .polar-series-area.inactive,.ngx-charts .area-series.inactive{transition:opacity .1s ease-in-out;opacity:.2}.ngx-charts .line-highlight{display:none}.ngx-charts .line-highlight.active{display:block}.ngx-charts .area{opacity:.6}.ngx-charts .circle:hover{cursor:pointer}.ngx-charts .label{font-size:12px;font-weight:400}.ngx-charts .tooltip-anchor{fill:#000}.ngx-charts .gridline-path{stroke:#ddd;stroke-width:1;fill:none}.ngx-charts .refline-path{stroke:#a8b2c7;stroke-width:1;stroke-dasharray:5;stroke-dashoffset:5}.ngx-charts .refline-label{font-size:9px}.ngx-charts .reference-area{fill-opacity:.05;fill:#000}.ngx-charts .gridline-path-dotted{stroke:#ddd;stroke-width:1;fill:none;stroke-dasharray:1,20;stroke-dashoffset:3}.ngx-charts .grid-panel rect{fill:none}.ngx-charts .grid-panel.odd rect{fill:#0000000d}\n"], components: [{ type: i1.ChartComponent, selector: "ngx-charts-chart", inputs: ["view", "showLegend", "legendOptions", "legendType", "activeEntries", "animations"], outputs: ["legendLabelClick", "legendLabelActivate", "legendLabelDeactivate"] }], directives: [{ type: i2.NgForOf, selector: "[ngFor][ngForOf]", inputs: ["ngForOf", "ngForTrackBy", "ngForTemplate"] }, { type: i3.TooltipDirective, selector: "[ngx-tooltip]", inputs: ["tooltipCssClass", "tooltipTitle", "tooltipAppendToBody", "tooltipSpacing", "tooltipDisabled", "tooltipShowCaret", "tooltipPlacement", "tooltipAlignment", "tooltipType", "tooltipCloseOnClickOutside", "tooltipCloseOnMouseLeave", "tooltipHideTimeout", "tooltipShowTimeout", "tooltipTemplate", "tooltipShowEvent", "tooltipContext", "tooltipImmediateExit"], outputs: ["show", "hide"] }, { type: i2.NgIf, selector: "[ngIf]", inputs: ["ngIf", "ngIfThen", "ngIfElse"] }], changeDetection: i0.ChangeDetectionStrategy.OnPush, encapsulation: i0.ViewEncapsulation.None });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "13.1.0", ngImport: i0, type: SankeyComponent, decorators: [{
            type: Component,
            args: [{ selector: 'ngx-charts-sankey', template: `
    <ngx-charts-chart [view]="[width, height]" [animations]="animations">
      <svg:g [attr.transform]="transform" class="sankey chart">
        <svg:g
          *ngFor="let link of linkPaths"
          class="link"
          ngx-tooltip
          [tooltipDisabled]="tooltipDisabled"
          [tooltipType]="styleTypes.tooltip"
          [tooltipPlacement]="'top'"
          [tooltipTitle]="tooltipTemplate ? undefined : link.tooltip"
          [tooltipTemplate]="tooltipTemplate"
          [tooltipContext]="link.data"
        >
          <svg:defs>
            <svg:linearGradient
              [attr.id]="link.id"
              gradientUnits="userSpaceOnUse"
              [attr.x1]="link.source.x1"
              [attr.x2]="link.target.x0"
            >
              <svg:stop offset="0%" [attr.stop-color]="link.startColor"></svg:stop>
              <svg:stop offset="100%" [attr.stop-color]="link.endColor"></svg:stop>
            </svg:linearGradient>
          </svg:defs>
          <svg:path
            [attr.d]="link.path"
            [attr.stroke]="link.gradientFill"
            [attr.stroke-width]="link.strokeWidth"
            stroke-opacity="0.5"
            fill="none"
            (click)="select.emit(link.data)"
            (mouseenter)="activate.emit(link.data)"
          ></svg:path>
        </svg:g>

        <svg:g *ngFor="let rect of nodeRects" [attr.transform]="rect.transform" class="node">
          <svg:rect
            [attr.x]="0"
            [attr.y]="0"
            [attr.width]="rect.width"
            [attr.height]="rect.height"
            [attr.fill]="rect.fill"
            ngx-tooltip
            [tooltipDisabled]="tooltipDisabled"
            [tooltipType]="styleTypes.tooltip"
            [tooltipPlacement]="'top'"
            [tooltipTitle]="tooltipTemplate ? undefined : rect.tooltip"
            [tooltipTemplate]="tooltipTemplate"
            [tooltipContext]="rect.data"
            (click)="select.emit(rect.data)"
            (mouseenter)="activate.emit(rect.data)"
          ></svg:rect>
        </svg:g>

        <svg:g *ngFor="let rect of nodeRects" [attr.transform]="rect.transform">
          <svg:text
            *ngIf="showLabels && rect.height > 15"
            class="label"
            [attr.x]="rect.width + 5"
            [attr.y]="rect.height / 2"
            [attr.text-anchor]="rect.labelAnchor"
            dy="0.35em"
            [attr.dx]="rect.labelAnchor === 'end' ? -25 : 0"
          >
            {{ rect.label }}
          </svg:text>
        </svg:g>
      </svg:g>
    </ngx-charts-chart>
  `, changeDetection: ChangeDetectionStrategy.OnPush, encapsulation: ViewEncapsulation.None, styles: [".ngx-charts-outer{animation:chartFadeIn linear .6s}@keyframes chartFadeIn{0%{opacity:0}20%{opacity:0}to{opacity:1}}.ngx-charts{float:left;overflow:visible}.ngx-charts .circle,.ngx-charts .cell,.ngx-charts .bar,.ngx-charts .node,.ngx-charts .link,.ngx-charts .arc{cursor:pointer}.ngx-charts .bar.active,.ngx-charts .bar:hover,.ngx-charts .cell.active,.ngx-charts .cell:hover,.ngx-charts .arc.active,.ngx-charts .arc:hover,.ngx-charts .node.active,.ngx-charts .node:hover,.ngx-charts .link.active,.ngx-charts .link:hover,.ngx-charts .card.active,.ngx-charts .card:hover{opacity:.8;transition:opacity .1s ease-in-out}.ngx-charts .bar:focus,.ngx-charts .cell:focus,.ngx-charts .arc:focus,.ngx-charts .node:focus,.ngx-charts .link:focus,.ngx-charts .card:focus{outline:none}.ngx-charts .bar.hidden,.ngx-charts .cell.hidden,.ngx-charts .arc.hidden,.ngx-charts .node.hidden,.ngx-charts .link.hidden,.ngx-charts .card.hidden{display:none}.ngx-charts g:focus{outline:none}.ngx-charts .line-series.inactive,.ngx-charts .line-series-range.inactive,.ngx-charts .polar-series-path.inactive,.ngx-charts .polar-series-area.inactive,.ngx-charts .area-series.inactive{transition:opacity .1s ease-in-out;opacity:.2}.ngx-charts .line-highlight{display:none}.ngx-charts .line-highlight.active{display:block}.ngx-charts .area{opacity:.6}.ngx-charts .circle:hover{cursor:pointer}.ngx-charts .label{font-size:12px;font-weight:400}.ngx-charts .tooltip-anchor{fill:#000}.ngx-charts .gridline-path{stroke:#ddd;stroke-width:1;fill:none}.ngx-charts .refline-path{stroke:#a8b2c7;stroke-width:1;stroke-dasharray:5;stroke-dashoffset:5}.ngx-charts .refline-label{font-size:9px}.ngx-charts .reference-area{fill-opacity:.05;fill:#000}.ngx-charts .gridline-path-dotted{stroke:#ddd;stroke-width:1;fill:none;stroke-dasharray:1,20;stroke-dashoffset:3}.ngx-charts .grid-panel rect{fill:none}.ngx-charts .grid-panel.odd rect{fill:#0000000d}\n"] }]
        }], propDecorators: { showLabels: [{
                type: Input
            }], gradient: [{
                type: Input
            }], tooltipDisabled: [{
                type: Input
            }], activeEntries: [{
                type: Input
            }], labelFormatting: [{
                type: Input
            }], activate: [{
                type: Output
            }], deactivate: [{
                type: Output
            }], tooltipTemplate: [{
                type: ContentChild,
                args: ['tooltipTemplate']
            }] } });
//# sourceMappingURL=data:application/json;base64,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