/* eslint-disable @typescript-eslint/no-parameter-properties */
import { isPlatformBrowser, CommonModule } from '@angular/common';
import { Component, forwardRef, Inject, Input, PLATFORM_ID, InjectionToken, Optional, ChangeDetectionStrategy } from '@angular/core';
import { FormsModule, NG_VALUE_ACCESSOR } from '@angular/forms';
import { Subject, takeUntil } from 'rxjs';
import { getTinymce } from '../TinyMCE';
import { listenTinyMCEEvent, bindHandlers, isTextarea, mergePlugins, uuid, noop, isNullOrUndefined } from '../utils/Utils';
import { Events } from './Events';
import { ScriptLoader } from '../utils/ScriptLoader';
import * as i0 from "@angular/core";
export const TINYMCE_SCRIPT_SRC = new InjectionToken('TINYMCE_SCRIPT_SRC');
const EDITOR_COMPONENT_VALUE_ACCESSOR = {
    provide: NG_VALUE_ACCESSOR,
    useExisting: forwardRef(() => EditorComponent),
    multi: true
};
/**
 * @see {@link https://www.tiny.cloud/docs/tinymce/7/angular-ref/} for the TinyMCE Angular Technical Reference
 */
export class EditorComponent extends Events {
    cdRef;
    platformId;
    tinymceScriptSrc;
    cloudChannel = '7';
    apiKey = 'no-api-key';
    licenseKey;
    init;
    id = '';
    initialValue;
    outputFormat;
    inline;
    tagName;
    plugins;
    toolbar;
    modelEvents = 'change input undo redo';
    allowedEvents;
    ignoreEvents;
    set disabled(val) {
        this._disabled = val;
        if (this._editor && this._editor.initialized) {
            if (typeof this._editor.mode?.set === 'function') {
                this._editor.mode.set(val ? 'readonly' : 'design');
            }
            else {
                this._editor.setMode(val ? 'readonly' : 'design');
            }
        }
    }
    get disabled() {
        return this._disabled;
    }
    get editor() {
        return this._editor;
    }
    ngZone;
    _elementRef;
    _element;
    _disabled;
    _editor;
    onTouchedCallback = noop;
    onChangeCallback;
    destroy$ = new Subject();
    constructor(elementRef, ngZone, cdRef, platformId, tinymceScriptSrc) {
        super();
        this.cdRef = cdRef;
        this.platformId = platformId;
        this.tinymceScriptSrc = tinymceScriptSrc;
        this._elementRef = elementRef;
        this.ngZone = ngZone;
    }
    writeValue(value) {
        if (this._editor && this._editor.initialized) {
            this._editor.setContent(isNullOrUndefined(value) ? '' : value);
        }
        else {
            this.initialValue = value === null ? undefined : value;
        }
    }
    registerOnChange(fn) {
        this.onChangeCallback = fn;
    }
    registerOnTouched(fn) {
        this.onTouchedCallback = fn;
    }
    setDisabledState(isDisabled) {
        this.disabled = isDisabled;
    }
    ngAfterViewInit() {
        if (isPlatformBrowser(this.platformId)) {
            this.id = this.id || uuid('tiny-angular');
            this.inline = this.inline !== undefined ? this.inline !== false : !!(this.init?.inline);
            this.createElement();
            if (getTinymce() !== null) {
                this.initialise();
            }
            else if (this._element && this._element.ownerDocument) {
                // Caretaker note: the component might be destroyed before the script is loaded and its code is executed.
                // This will lead to runtime exceptions if `initialise` will be called when the component has been destroyed.
                ScriptLoader.load(this._element.ownerDocument, this.getScriptSrc())
                    .pipe(takeUntil(this.destroy$))
                    .subscribe(this.initialise);
            }
        }
    }
    ngOnDestroy() {
        this.destroy$.next();
        if (getTinymce() !== null) {
            getTinymce().remove(this._editor);
        }
    }
    createElement() {
        const tagName = typeof this.tagName === 'string' ? this.tagName : 'div';
        this._element = document.createElement(this.inline ? tagName : 'textarea');
        if (this._element) {
            const existingElement = document.getElementById(this.id);
            if (existingElement && existingElement !== this._elementRef.nativeElement) {
                /* eslint no-console: ["error", { allow: ["warn"] }] */
                console.warn(`TinyMCE-Angular: an element with id [${this.id}] already exists. Editors with duplicate Id will not be able to mount`);
            }
            this._element.id = this.id;
            if (isTextarea(this._element)) {
                this._element.style.visibility = 'hidden';
            }
            this._elementRef.nativeElement.appendChild(this._element);
        }
    }
    initialise = () => {
        const finalInit = {
            ...this.init,
            selector: undefined,
            target: this._element,
            inline: this.inline,
            readonly: this.disabled,
            license_key: this.licenseKey,
            plugins: mergePlugins((this.init && this.init.plugins), this.plugins),
            toolbar: this.toolbar || (this.init && this.init.toolbar),
            setup: (editor) => {
                this._editor = editor;
                listenTinyMCEEvent(editor, 'init', this.destroy$).subscribe(() => {
                    this.initEditor(editor);
                });
                bindHandlers(this, editor, this.destroy$);
                if (this.init && typeof this.init.setup === 'function') {
                    this.init.setup(editor);
                }
            }
        };
        if (isTextarea(this._element)) {
            this._element.style.visibility = '';
        }
        this.ngZone.runOutsideAngular(() => {
            getTinymce().init(finalInit);
        });
    };
    getScriptSrc() {
        return isNullOrUndefined(this.tinymceScriptSrc) ?
            `https://cdn.tiny.cloud/1/${this.apiKey}/tinymce/${this.cloudChannel}/tinymce.min.js` :
            this.tinymceScriptSrc;
    }
    initEditor(editor) {
        listenTinyMCEEvent(editor, 'blur', this.destroy$).subscribe(() => {
            this.cdRef.markForCheck();
            this.ngZone.run(() => this.onTouchedCallback());
        });
        listenTinyMCEEvent(editor, this.modelEvents, this.destroy$).subscribe(() => {
            this.cdRef.markForCheck();
            this.ngZone.run(() => this.emitOnChange(editor));
        });
        if (typeof this.initialValue === 'string') {
            this.ngZone.run(() => {
                editor.setContent(this.initialValue);
                if (editor.getContent() !== this.initialValue) {
                    this.emitOnChange(editor);
                }
                if (this.onInitNgModel !== undefined) {
                    this.onInitNgModel.emit(editor);
                }
            });
        }
    }
    emitOnChange(editor) {
        if (this.onChangeCallback) {
            this.onChangeCallback(editor.getContent({ format: this.outputFormat }));
        }
    }
    static ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "17.3.11", ngImport: i0, type: EditorComponent, deps: [{ token: i0.ElementRef }, { token: i0.NgZone }, { token: i0.ChangeDetectorRef }, { token: PLATFORM_ID }, { token: TINYMCE_SCRIPT_SRC, optional: true }], target: i0.ɵɵFactoryTarget.Component });
    static ɵcmp = i0.ɵɵngDeclareComponent({ minVersion: "14.0.0", version: "17.3.11", type: EditorComponent, isStandalone: true, selector: "editor", inputs: { cloudChannel: "cloudChannel", apiKey: "apiKey", licenseKey: "licenseKey", init: "init", id: "id", initialValue: "initialValue", outputFormat: "outputFormat", inline: "inline", tagName: "tagName", plugins: "plugins", toolbar: "toolbar", modelEvents: "modelEvents", allowedEvents: "allowedEvents", ignoreEvents: "ignoreEvents", disabled: "disabled" }, providers: [EDITOR_COMPONENT_VALUE_ACCESSOR], usesInheritance: true, ngImport: i0, template: '', isInline: true, styles: [":host{display:block}\n"], dependencies: [{ kind: "ngmodule", type: CommonModule }, { kind: "ngmodule", type: FormsModule }], changeDetection: i0.ChangeDetectionStrategy.OnPush });
}
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "17.3.11", ngImport: i0, type: EditorComponent, decorators: [{
            type: Component,
            args: [{ selector: 'editor', template: '', providers: [EDITOR_COMPONENT_VALUE_ACCESSOR], standalone: true, imports: [CommonModule, FormsModule], changeDetection: ChangeDetectionStrategy.OnPush, styles: [":host{display:block}\n"] }]
        }], ctorParameters: () => [{ type: i0.ElementRef }, { type: i0.NgZone }, { type: i0.ChangeDetectorRef }, { type: Object, decorators: [{
                    type: Inject,
                    args: [PLATFORM_ID]
                }] }, { type: undefined, decorators: [{
                    type: Optional
                }, {
                    type: Inject,
                    args: [TINYMCE_SCRIPT_SRC]
                }] }], propDecorators: { cloudChannel: [{
                type: Input
            }], apiKey: [{
                type: Input
            }], licenseKey: [{
                type: Input
            }], init: [{
                type: Input
            }], id: [{
                type: Input
            }], initialValue: [{
                type: Input
            }], outputFormat: [{
                type: Input
            }], inline: [{
                type: Input
            }], tagName: [{
                type: Input
            }], plugins: [{
                type: Input
            }], toolbar: [{
                type: Input
            }], modelEvents: [{
                type: Input
            }], allowedEvents: [{
                type: Input
            }], ignoreEvents: [{
                type: Input
            }], disabled: [{
                type: Input
            }] } });
//# sourceMappingURL=data:application/json;base64,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