/**
 * Copyright (c) 2017-present, Ephox, Inc.
 *
 * This source code is licensed under the Apache 2 license found in the
 * LICENSE file in the root directory of this source tree.
 *
 */
import { fromEvent, shareReplay, switchMap, BehaviorSubject, first, filter, map } from 'rxjs';
const firstEmission = () => (source) => source.pipe(first(), map(() => undefined));
const CreateScriptLoader = () => {
    const params$ = new BehaviorSubject(null);
    const loaded$ = params$.pipe(filter(Boolean), switchMap(([doc, url]) => {
        const scriptTag = doc.createElement('script');
        scriptTag.referrerPolicy = 'origin';
        scriptTag.type = 'application/javascript';
        scriptTag.src = url;
        doc.head.appendChild(scriptTag);
        return fromEvent(scriptTag, 'load').pipe(firstEmission());
    }), 
    // Caretaker note: `loaded$` is a multicast observable since it's piped with `shareReplay`,
    // so if there're multiple editor components simultaneously on the page, they'll subscribe to the internal
    // `ReplaySubject`. The script will be loaded only once, and `ReplaySubject` will cache the result.
    shareReplay({ bufferSize: 1, refCount: true }));
    return {
        load: (...args) => {
            if (!params$.getValue()) {
                params$.next(args);
            }
            return loaded$;
        },
        reinitialize: () => {
            params$.next(null);
        },
    };
};
const ScriptLoader = CreateScriptLoader();
export { ScriptLoader };
//# sourceMappingURL=data:application/json;base64,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