/**
 * Copyright (c) 2017-present, Ephox, Inc.
 *
 * This source code is licensed under the Apache 2 license found in the
 * LICENSE file in the root directory of this source tree.
 *
 */
import { fromEvent, takeUntil } from 'rxjs';
import { validEvents } from '../editor/Events';
// Caretaker note: `fromEvent` supports passing JQuery-style event targets, the editor has `on` and `off` methods which
// will be invoked upon subscription and teardown.
const listenTinyMCEEvent = (editor, eventName, destroy$) => fromEvent(editor, eventName).pipe(takeUntil(destroy$));
const bindHandlers = (ctx, editor, destroy$) => {
    const allowedEvents = getValidEvents(ctx);
    allowedEvents.forEach((eventName) => {
        const eventEmitter = ctx[eventName];
        listenTinyMCEEvent(editor, eventName.substring(2), destroy$).subscribe((event) => {
            // Caretaker note: `ngZone.run()` runs change detection since it notifies the forked Angular zone that it's
            // being re-entered. We don't want to run `ApplicationRef.tick()` if anyone listens to the specific event
            // within the template. E.g. if the `onSelectionChange` is not listened within the template like:
            // `<editor (onSelectionChange)="..."></editor>`
            // then it won't be "observed", and we won't run "dead" change detection.
            if (isObserved(eventEmitter)) {
                ctx.ngZone.run(() => eventEmitter.emit({ event, editor }));
            }
        });
    });
};
const getValidEvents = (ctx) => {
    const ignoredEvents = parseStringProperty(ctx.ignoreEvents, []);
    const allowedEvents = parseStringProperty(ctx.allowedEvents, validEvents).filter((event) => validEvents.includes(event) && !ignoredEvents.includes(event));
    return allowedEvents;
};
const parseStringProperty = (property, defaultValue) => {
    if (typeof property === 'string') {
        return property.split(',').map((value) => value.trim());
    }
    if (Array.isArray(property)) {
        return property;
    }
    return defaultValue;
};
let unique = 0;
const uuid = (prefix) => {
    const date = new Date();
    const time = date.getTime();
    const random = Math.floor(Math.random() * 1000000000);
    unique++;
    return prefix + '_' + random + unique + String(time);
};
const isTextarea = (element) => typeof element !== 'undefined' && element.tagName.toLowerCase() === 'textarea';
const normalizePluginArray = (plugins) => {
    if (typeof plugins === 'undefined' || plugins === '') {
        return [];
    }
    return Array.isArray(plugins) ? plugins : plugins.split(' ');
};
const mergePlugins = (initPlugins, inputPlugins) => normalizePluginArray(initPlugins).concat(normalizePluginArray(inputPlugins));
// eslint-disable-next-line @typescript-eslint/no-empty-function
const noop = () => { };
const isNullOrUndefined = (value) => value === null || value === undefined;
const isObserved = (o) => 
// RXJS is making the `observers` property internal in v8. So this is intended as a backwards compatible way of
// checking if a subject has observers.
o.observed || o.observers?.length > 0;
export { listenTinyMCEEvent, bindHandlers, uuid, isTextarea, normalizePluginArray, mergePlugins, noop, isNullOrUndefined };
//# sourceMappingURL=data:application/json;base64,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