import { Inject, Injectable, InjectionToken, Optional, PLATFORM_ID, SkipSelf } from '@angular/core';
import { of as observableOf, throwError as observableThrowError } from 'rxjs';
import { catchError, finalize, map, share, tap } from 'rxjs/operators';
import { DOCUMENT } from '@angular/common';
import { SvgLoader } from './svg-loader';
import * as i0 from "@angular/core";
import * as i1 from "./svg-loader";
export const SERVER_URL = new InjectionToken('SERVER_URL');
export class SvgIconRegistryService {
    constructor(loader, platformId, serverUrl, _document) {
        this.loader = loader;
        this.platformId = platformId;
        this.serverUrl = serverUrl;
        this._document = _document;
        this.iconsByUrl = new Map();
        this.iconsLoadingByUrl = new Map();
        this.document = this._document;
    }
    /** Add a SVG to the registry by passing a name and the SVG. */
    addSvg(name, data) {
        if (!this.iconsByUrl.has(name)) {
            const div = this.document.createElement('DIV');
            div.innerHTML = data;
            const svg = div.querySelector('svg');
            this.iconsByUrl.set(name, svg);
        }
    }
    /** Load a SVG to the registry from a URL. */
    loadSvg(url, name = url) {
        // not sure if there should be a possibility to use name for server usage
        // so overriding it for now if provided
        // maybe should separate functionality for url and name use-cases
        if (this.serverUrl && url.match(/^(http(s)?):/) === null) {
            url = this.serverUrl + url;
            name = url;
        }
        if (this.iconsByUrl.has(name)) {
            return observableOf(this.iconsByUrl.get(name));
        }
        else if (this.iconsLoadingByUrl.has(name)) {
            return this.iconsLoadingByUrl.get(name);
        }
        const o = this.loader.getSvg(url).pipe(map(svg => {
            const div = this.document.createElement('DIV');
            div.innerHTML = svg;
            return div.querySelector('svg');
        }), tap(svg => this.iconsByUrl.set(name, svg)), catchError(err => {
            console.error(err);
            return observableThrowError(err);
        }), finalize(() => this.iconsLoadingByUrl.delete(name)), share());
        this.iconsLoadingByUrl.set(name, o);
        return o;
    }
    /** Get loaded SVG from registry by name. (also works by url because of blended map) */
    getSvgByName(name) {
        if (this.iconsByUrl.has(name)) {
            return observableOf(this.iconsByUrl.get(name));
        }
        else if (this.iconsLoadingByUrl.has(name)) {
            return this.iconsLoadingByUrl.get(name);
        }
        return observableThrowError(`No svg with name '${name}' has been loaded`);
    }
    /** Remove a SVG from the registry by URL (or name). */
    unloadSvg(url) {
        if (this.iconsByUrl.has(url)) {
            this.iconsByUrl.delete(url);
        }
    }
    static { this.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "17.0.2", ngImport: i0, type: SvgIconRegistryService, deps: [{ token: i1.SvgLoader }, { token: PLATFORM_ID }, { token: SERVER_URL, optional: true }, { token: DOCUMENT, optional: true }], target: i0.ɵɵFactoryTarget.Injectable }); }
    static { this.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "17.0.2", ngImport: i0, type: SvgIconRegistryService }); }
}
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "17.0.2", ngImport: i0, type: SvgIconRegistryService, decorators: [{
            type: Injectable
        }], ctorParameters: () => [{ type: i1.SvgLoader }, { type: Object, decorators: [{
                    type: Inject,
                    args: [PLATFORM_ID]
                }] }, { type: undefined, decorators: [{
                    type: Optional
                }, {
                    type: Inject,
                    args: [SERVER_URL]
                }] }, { type: undefined, decorators: [{
                    type: Optional
                }, {
                    type: Inject,
                    args: [DOCUMENT]
                }] }] });
export function SVG_ICON_REGISTRY_PROVIDER_FACTORY(parentRegistry, loader, platformId, serverUrl, document) {
    return parentRegistry || new SvgIconRegistryService(loader, platformId, serverUrl, document);
}
export const SVG_ICON_REGISTRY_PROVIDER = {
    provide: SvgIconRegistryService,
    deps: [[new Optional(), new SkipSelf(), SvgIconRegistryService], SvgLoader, [PLATFORM_ID],
        [new Optional(), SERVER_URL], [new Optional(), DOCUMENT]
    ],
    useFactory: SVG_ICON_REGISTRY_PROVIDER_FACTORY
};
//# sourceMappingURL=data:application/json;base64,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