import { from } from 'rxjs';
import { Inject, Injectable, Optional } from '@angular/core';
import { ApolloClient } from '@apollo/client/core';
import { QueryRef } from './query-ref';
import { APOLLO_FLAGS, APOLLO_NAMED_OPTIONS, APOLLO_OPTIONS } from './tokens';
import { fixObservable, fromPromise, useMutationLoading, wrapWithZone } from './utils';
import * as i0 from "@angular/core";
export class ApolloBase {
    ngZone;
    flags;
    _client;
    useInitialLoading;
    useMutationLoading;
    constructor(ngZone, flags, _client) {
        this.ngZone = ngZone;
        this.flags = flags;
        this._client = _client;
        this.useInitialLoading = flags?.useInitialLoading ?? false;
        this.useMutationLoading = flags?.useMutationLoading ?? false;
    }
    watchQuery(options) {
        return new QueryRef(this.ensureClient().watchQuery({
            ...options,
        }), this.ngZone, {
            useInitialLoading: this.useInitialLoading,
            ...options,
        });
    }
    query(options) {
        return fromPromise(() => this.ensureClient().query({ ...options }));
    }
    mutate(options) {
        return useMutationLoading(fromPromise(() => this.ensureClient().mutate({ ...options })), options.useMutationLoading ?? this.useMutationLoading);
    }
    watchFragment(options, extra) {
        const obs = from(fixObservable(this.ensureClient().watchFragment({ ...options })));
        return extra && extra.useZone !== true ? obs : wrapWithZone(obs, this.ngZone);
    }
    subscribe(options, extra) {
        const obs = from(fixObservable(this.ensureClient().subscribe({ ...options })));
        return extra && extra.useZone !== true ? obs : wrapWithZone(obs, this.ngZone);
    }
    /**
     * Get an instance of ApolloClient
     */
    get client() {
        return this.ensureClient();
    }
    /**
     * Set a new instance of ApolloClient
     * Remember to clean up the store before setting a new client.
     *
     * @param client ApolloClient instance
     */
    set client(client) {
        if (this._client) {
            throw new Error('Client has been already defined');
        }
        this._client = client;
    }
    ensureClient() {
        this.checkInstance();
        return this._client;
    }
    checkInstance() {
        if (this._client) {
            return true;
        }
        else {
            throw new Error('Client has not been defined yet');
        }
    }
}
export class Apollo extends ApolloBase {
    map = new Map();
    constructor(ngZone, apolloOptions, apolloNamedOptions, flags) {
        super(ngZone, flags);
        if (apolloOptions) {
            this.createDefault(apolloOptions);
        }
        if (apolloNamedOptions && typeof apolloNamedOptions === 'object') {
            for (let name in apolloNamedOptions) {
                if (apolloNamedOptions.hasOwnProperty(name)) {
                    const options = apolloNamedOptions[name];
                    this.create(options, name);
                }
            }
        }
    }
    /**
     * Create an instance of ApolloClient
     * @param options Options required to create ApolloClient
     * @param name client's name
     */
    create(options, name) {
        if (isNamed(name)) {
            this.createNamed(name, options);
        }
        else {
            this.createDefault(options);
        }
    }
    /**
     * Use a default ApolloClient
     */
    default() {
        return this;
    }
    /**
     * Use a named ApolloClient
     * @param name client's name
     */
    use(name) {
        if (isNamed(name)) {
            return this.map.get(name);
        }
        else {
            return this.default();
        }
    }
    /**
     * Create a default ApolloClient, same as `apollo.create(options)`
     * @param options ApolloClient's options
     */
    createDefault(options) {
        if (this._client) {
            throw new Error('Apollo has been already created.');
        }
        this.client = this.ngZone.runOutsideAngular(() => new ApolloClient(options));
    }
    /**
     * Create a named ApolloClient, same as `apollo.create(options, name)`
     * @param name client's name
     * @param options ApolloClient's options
     */
    createNamed(name, options) {
        if (this.map.has(name)) {
            throw new Error(`Client ${name} has been already created`);
        }
        this.map.set(name, new ApolloBase(this.ngZone, this.flags, this.ngZone.runOutsideAngular(() => new ApolloClient(options))));
    }
    /**
     * Remember to clean up the store before removing a client
     * @param name client's name
     */
    removeClient(name) {
        if (isNamed(name)) {
            this.map.delete(name);
        }
        else {
            this._client = undefined;
        }
    }
    static ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "18.0.5", ngImport: i0, type: Apollo, deps: [{ token: i0.NgZone }, { token: APOLLO_OPTIONS, optional: true }, { token: APOLLO_NAMED_OPTIONS, optional: true }, { token: APOLLO_FLAGS, optional: true }], target: i0.ɵɵFactoryTarget.Injectable });
    static ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "18.0.5", ngImport: i0, type: Apollo });
}
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "18.0.5", ngImport: i0, type: Apollo, decorators: [{
            type: Injectable
        }], ctorParameters: () => [{ type: i0.NgZone }, { type: undefined, decorators: [{
                    type: Optional
                }, {
                    type: Inject,
                    args: [APOLLO_OPTIONS]
                }] }, { type: undefined, decorators: [{
                    type: Inject,
                    args: [APOLLO_NAMED_OPTIONS]
                }, {
                    type: Optional
                }] }, { type: undefined, decorators: [{
                    type: Inject,
                    args: [APOLLO_FLAGS]
                }, {
                    type: Optional
                }] }] });
function isNamed(name) {
    return !!name && name !== 'default';
}
//# sourceMappingURL=data:application/json;base64,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