import { print } from 'graphql';
import { Injectable } from '@angular/core';
import { ApolloLink, Observable as LinkObservable, } from '@apollo/client/core';
import { BatchLink } from '@apollo/client/link/batch';
import { createHeadersWithClientAwareness, fetch, mergeHeaders, prioritize } from './utils';
import * as i0 from "@angular/core";
import * as i1 from "@angular/common/http";
export const defaults = {
    batchInterval: 10,
    batchMax: 10,
    uri: 'graphql',
    method: 'POST',
    withCredentials: false,
    includeQuery: true,
    includeExtensions: false,
    useMultipart: false,
};
/**
 * Decides which value to pick from Context, Options or defaults
 */
export function pick(context, options, key) {
    return prioritize(context[key], options[key], defaults[key]);
}
export class HttpBatchLinkHandler extends ApolloLink {
    httpClient;
    options;
    batcher;
    batchInterval;
    batchMax;
    print = print;
    constructor(httpClient, options) {
        super();
        this.httpClient = httpClient;
        this.options = options;
        this.batchInterval = options.batchInterval || defaults.batchInterval;
        this.batchMax = options.batchMax || defaults.batchMax;
        if (this.options.operationPrinter) {
            this.print = this.options.operationPrinter;
        }
        const batchHandler = (operations) => {
            return new LinkObservable((observer) => {
                const body = this.createBody(operations);
                const headers = this.createHeaders(operations);
                const { method, uri, withCredentials } = this.createOptions(operations);
                if (typeof uri === 'function') {
                    throw new Error(`Option 'uri' is a function, should be a string`);
                }
                const req = {
                    method,
                    url: uri,
                    body: body,
                    options: {
                        withCredentials,
                        headers,
                    },
                };
                const sub = fetch(req, this.httpClient, () => {
                    throw new Error('File upload is not available when combined with Batching');
                }).subscribe({
                    next: result => observer.next(result.body),
                    error: err => observer.error(err),
                    complete: () => observer.complete(),
                });
                return () => {
                    if (!sub.closed) {
                        sub.unsubscribe();
                    }
                };
            });
        };
        const batchKey = options.batchKey ||
            ((operation) => {
                return this.createBatchKey(operation);
            });
        this.batcher = new BatchLink({
            batchInterval: this.batchInterval,
            batchMax: this.batchMax,
            batchKey,
            batchHandler,
        });
    }
    createOptions(operations) {
        const context = operations[0].getContext();
        return {
            method: pick(context, this.options, 'method'),
            uri: pick(context, this.options, 'uri'),
            withCredentials: pick(context, this.options, 'withCredentials'),
        };
    }
    createBody(operations) {
        return operations.map(operation => {
            const includeExtensions = prioritize(operation.getContext().includeExtensions, this.options.includeExtensions, false);
            const includeQuery = prioritize(operation.getContext().includeQuery, this.options.includeQuery, true);
            const body = {
                operationName: operation.operationName,
                variables: operation.variables,
            };
            if (includeExtensions) {
                body.extensions = operation.extensions;
            }
            if (includeQuery) {
                body.query = this.print(operation.query);
            }
            return body;
        });
    }
    createHeaders(operations) {
        return operations.reduce((headers, operation) => {
            return mergeHeaders(headers, operation.getContext().headers);
        }, createHeadersWithClientAwareness({
            headers: this.options.headers,
            clientAwareness: operations[0]?.getContext()?.clientAwareness,
        }));
    }
    createBatchKey(operation) {
        const context = operation.getContext();
        if (context.skipBatching) {
            return Math.random().toString(36).substr(2, 9);
        }
        const headers = context.headers && context.headers.keys().map((k) => context.headers.get(k));
        const opts = JSON.stringify({
            includeQuery: context.includeQuery,
            includeExtensions: context.includeExtensions,
            headers,
        });
        return prioritize(context.uri, this.options.uri, '') + opts;
    }
    request(op) {
        return this.batcher.request(op);
    }
}
export class HttpBatchLink {
    httpClient;
    constructor(httpClient) {
        this.httpClient = httpClient;
    }
    create(options) {
        return new HttpBatchLinkHandler(this.httpClient, options);
    }
    static ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "18.0.5", ngImport: i0, type: HttpBatchLink, deps: [{ token: i1.HttpClient }], target: i0.ɵɵFactoryTarget.Injectable });
    static ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "18.0.5", ngImport: i0, type: HttpBatchLink, providedIn: 'root' });
}
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "18.0.5", ngImport: i0, type: HttpBatchLink, decorators: [{
            type: Injectable,
            args: [{
                    providedIn: 'root',
                }]
        }], ctorParameters: () => [{ type: i1.HttpClient }] });
//# sourceMappingURL=data:application/json;base64,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