import { Observable } from 'rxjs';
import { HttpHeaders } from '@angular/common/http';
export const fetch = (req, httpClient, extractFiles) => {
    const shouldUseBody = ['POST', 'PUT', 'PATCH'].indexOf(req.method.toUpperCase()) !== -1;
    const shouldStringify = (param) => ['variables', 'extensions'].indexOf(param.toLowerCase()) !== -1;
    const isBatching = req.body.length;
    let shouldUseMultipart = req.options && req.options.useMultipart;
    let multipartInfo;
    if (shouldUseMultipart) {
        if (isBatching) {
            return new Observable(observer => observer.error(new Error('File upload is not available when combined with Batching')));
        }
        if (!shouldUseBody) {
            return new Observable(observer => observer.error(new Error('File upload is not available when GET is used')));
        }
        if (!extractFiles) {
            return new Observable(observer => observer.error(new Error(`To use File upload you need to pass "extractFiles" function from "extract-files" library to HttpLink's options`)));
        }
        multipartInfo = extractFiles(req.body);
        shouldUseMultipart = !!multipartInfo.files.size;
    }
    // `body` for some, `params` for others
    let bodyOrParams = {};
    if (isBatching) {
        if (!shouldUseBody) {
            return new Observable(observer => observer.error(new Error('Batching is not available for GET requests')));
        }
        bodyOrParams = {
            body: req.body,
        };
    }
    else {
        const body = shouldUseMultipart ? multipartInfo.clone : req.body;
        if (shouldUseBody) {
            bodyOrParams = {
                body,
            };
        }
        else {
            const params = Object.keys(req.body).reduce((obj, param) => {
                const value = req.body[param];
                obj[param] = shouldStringify(param) ? JSON.stringify(value) : value;
                return obj;
            }, {});
            bodyOrParams = { params: params };
        }
    }
    if (shouldUseMultipart && shouldUseBody) {
        const form = new FormData();
        form.append('operations', JSON.stringify(bodyOrParams.body));
        const map = {};
        const files = multipartInfo.files;
        let i = 0;
        files.forEach(paths => {
            map[++i] = paths;
        });
        form.append('map', JSON.stringify(map));
        i = 0;
        files.forEach((_, file) => {
            form.append(++i + '', file, file.name);
        });
        bodyOrParams.body = form;
    }
    // create a request
    return httpClient.request(req.method, req.url, {
        observe: 'response',
        responseType: 'json',
        reportProgress: false,
        ...bodyOrParams,
        ...req.options,
    });
};
export const mergeHeaders = (source, destination) => {
    if (source && destination) {
        const merged = destination
            .keys()
            .reduce((headers, name) => headers.set(name, destination.getAll(name)), source);
        return merged;
    }
    return destination || source;
};
export function prioritize(...values) {
    return values.find(val => typeof val !== 'undefined');
}
export function createHeadersWithClientAwareness(context) {
    // `apollographql-client-*` headers are automatically set if a
    // `clientAwareness` object is found in the context. These headers are
    // set first, followed by the rest of the headers pulled from
    // `context.headers`.
    let headers = context.headers && context.headers instanceof HttpHeaders
        ? context.headers
        : new HttpHeaders(context.headers);
    if (context.clientAwareness) {
        const { name, version } = context.clientAwareness;
        // If desired, `apollographql-client-*` headers set by
        // the `clientAwareness` object can be overridden by
        // `apollographql-client-*` headers set in `context.headers`.
        if (name && !headers.has('apollographql-client-name')) {
            headers = headers.set('apollographql-client-name', name);
        }
        if (version && !headers.has('apollographql-client-version')) {
            headers = headers.set('apollographql-client-version', version);
        }
    }
    return headers;
}
//# sourceMappingURL=data:application/json;base64,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