import { Component, ContentChildren, Input } from '@angular/core';
import { trigger, style, transition, animate, keyframes, state } from '@angular/animations';
import { ACTIONS, arrowKeysHandler } from './actions';
import { Ng2MenuItem } from '../menu-item/ng2-menu-item';
import * as i0 from "@angular/core";
import * as i1 from "../../services/dropdown-state.service";
import * as i2 from "@angular/common";
export class Ng2DropdownMenu {
    constructor(dropdownState, element, renderer) {
        this.dropdownState = dropdownState;
        this.element = element;
        this.renderer = renderer;
        /**
         * @name width
         */
        this.width = 4;
        /**
         * @description if set to true, the first element of the dropdown will be automatically focused
         * @name focusFirstElement
         */
        this.focusFirstElement = true;
        /**
         * @name appendToBody
         */
        this.appendToBody = true;
        /**
         * @name zIndex
         */
        this.zIndex = 1000;
        this.listeners = {
            arrowHandler: undefined,
            handleKeypress: undefined
        };
    }
    /**
     * @name show
     * @shows menu and selects first item
     */
    show(position, dynamic = true) {
        const dc = typeof document !== 'undefined' ? document : undefined;
        const wd = typeof window !== 'undefined' ? window : undefined;
        if (!this.dropdownState.menuState.isVisible) {
            // setting handlers
            this.listeners.handleKeypress = this.renderer.listen(dc.body, 'keydown', this.handleKeypress.bind(this));
            this.listeners.arrowHandler = this.renderer.listen(wd, 'keydown', arrowKeysHandler);
        }
        // update state
        this.dropdownState.menuState.isVisible = true;
        if (position) {
            this.updatePosition(position, dynamic);
        }
    }
    /**
     * @name hide
     * @desc hides menu
     */
    hide() {
        this.dropdownState.menuState.isVisible = false;
        // reset selected item state
        this.dropdownState.dropdownState.unselect();
        // call function to unlisten
        this.listeners.arrowHandler && this.listeners.arrowHandler();
        this.listeners.handleKeypress && this.listeners.handleKeypress();
    }
    /**
     * @name updatePosition
     * @desc updates the menu position every time it is toggled
     * @param position {ClientRect}
     * @param dynamic {boolean}
     */
    updatePosition(position, dynamic) {
        this.position = position;
        this.updateOnChange(dynamic);
    }
    /**
     * @name handleKeypress
     * @desc executes functions on keyPress based on the key pressed
     * @param $event
     */
    handleKeypress($event) {
        const key = $event.keyCode;
        const items = this.items.toArray();
        const index = items.indexOf(this.dropdownState.dropdownState.selectedItem);
        if (!ACTIONS.hasOwnProperty(key)) {
            return;
        }
        ACTIONS[key].call(this, index, items, this.dropdownState.dropdownState);
    }
    /**
     * @name getMenuElement
     */
    getMenuElement() {
        return this.element.nativeElement.children[0];
    }
    /**
     * @name calcPositionOffset
     * @param position
     */
    calcPositionOffset(position) {
        const wd = typeof window !== 'undefined' ? window : undefined;
        const dc = typeof document !== 'undefined' ? document : undefined;
        if (!wd || !dc || !position) {
            return;
        }
        const element = this.getMenuElement();
        const supportPageOffset = wd.pageXOffset !== undefined;
        const isCSS1Compat = (dc.compatMode || '') === 'CSS1Compat';
        const x = supportPageOffset
            ? wd.pageXOffset
            : isCSS1Compat
                ? dc.documentElement.scrollLeft
                : dc.body.scrollLeft;
        const y = supportPageOffset
            ? wd.pageYOffset
            : isCSS1Compat
                ? dc.documentElement.scrollTop
                : dc.body.scrollTop;
        let { top, left } = this.applyOffset(`${position.top + (this.appendToBody ? y - 15 : 0)}px`, `${position.left + x - 5}px`);
        const clientWidth = element.clientWidth;
        const clientHeight = element.clientHeight;
        const marginFromBottom = parseInt(top) + clientHeight + (this.appendToBody ? 0 : y - 15);
        const marginFromRight = parseInt(left) + clientWidth;
        const windowScrollHeight = wd.innerHeight + wd.scrollY;
        const windowScrollWidth = wd.innerWidth + wd.scrollX;
        if (marginFromBottom >= windowScrollHeight) {
            top = `${parseInt(top.replace('px', '')) - clientHeight}px`;
        }
        if (marginFromRight >= windowScrollWidth) {
            const marginRight = marginFromRight - windowScrollWidth + 30;
            left = `${parseInt(left.replace('px', '')) - marginRight}px`;
        }
        return { top, left };
    }
    applyOffset(top, left) {
        if (!this.offset) {
            return { top, left };
        }
        const offset = this.offset.split(' ');
        if (!offset[1]) {
            offset[1] = '0';
        }
        top = `${parseInt(top.replace('px', '')) + parseInt(offset[0])}px`;
        left = `${parseInt(left.replace('px', '')) + parseInt(offset[1])}px`;
        return { top, left };
    }
    ngOnInit() {
        const dc = typeof document !== 'undefined' ? document : undefined;
        if (this.appendToBody) {
            // append menu element to the body
            dc.body.appendChild(this.element.nativeElement);
        }
    }
    updateOnChange(dynamic = true) {
        const element = this.getMenuElement();
        const position = this.calcPositionOffset(this.position);
        if (position) {
            this.renderer.setStyle(element, 'top', position.top.toString());
            this.renderer.setStyle(element, 'left', position.left.toString());
        }
        // select first item unless user disabled this option
        if (this.focusFirstElement &&
            this.items.first &&
            !this.dropdownState.dropdownState.selectedItem) {
            this.dropdownState.dropdownState.select(this.items.first, false);
        }
    }
    ngOnDestroy() {
        const elem = this.element.nativeElement;
        elem.parentNode.removeChild(elem);
        if (this.listeners.handleKeypress) {
            this.listeners.handleKeypress();
        }
    }
}
Ng2DropdownMenu.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "13.0.3", ngImport: i0, type: Ng2DropdownMenu, deps: [{ token: i1.DropdownStateService }, { token: i0.ElementRef }, { token: i0.Renderer2 }], target: i0.ɵɵFactoryTarget.Component });
Ng2DropdownMenu.ɵcmp = i0.ɵɵngDeclareComponent({ minVersion: "12.0.0", version: "13.0.3", type: Ng2DropdownMenu, selector: "ng2-dropdown-menu", inputs: { width: "width", focusFirstElement: "focusFirstElement", offset: "offset", appendToBody: "appendToBody", zIndex: "zIndex" }, queries: [{ propertyName: "items", predicate: Ng2MenuItem, descendants: true }], ngImport: i0, template: `
        <!-- MENU -->
        <div
            class="ng2-dropdown-menu ng2-dropdown-menu---width--{{ width }}"
            [class.ng2-dropdown-menu--inside-element]="!appendToBody"
            [class.ng2-dropdown-menu--open]="dropdownState.menuState.isVisible"
            [style.z-index]="zIndex"
            [@fade]="dropdownState.menuState.toString()"
        >
            <div
                class="ng2-dropdown-menu__options-container"
                [@opacity]="dropdownState.menuState.toString()"
            >
                <ng-content></ng-content>
            </div>
        </div>

        <!-- BACKDROP -->
        <div
            class="ng2-dropdown-backdrop"
            *ngIf="dropdownState.menuState.isVisible"
            (click)="hide()"
        ></div>
    `, isInline: true, styles: [":host{display:block}.ng2-dropdown-menu{overflow-y:auto;box-shadow:0 1px 2px #0000004d;position:absolute;padding:.5em 0;background:#fff;border-radius:1px;max-height:400px;width:260px;min-height:0;display:block}.ng2-dropdown-menu.ng2-dropdown-menu--inside-element{position:fixed}.ng2-dropdown-menu.ng2-dropdown-menu--width--2{width:200px}.ng2-dropdown-menu.ng2-dropdown-menu--width--4{width:260px}.ng2-dropdown-menu.ng2-dropdown-menu--width--6{width:320px}.ng2-dropdown-backdrop{position:fixed;top:0;left:0;width:100%;height:100%;z-index:1;overflow:hidden}:host ::ng-deep .ng2-menu-divider{height:1px;min-height:1px;max-height:1px;width:100%;display:block;background:#f9f9f9}\n"], directives: [{ type: i2.NgIf, selector: "[ngIf]", inputs: ["ngIf", "ngIfThen", "ngIfElse"] }], animations: [
        trigger('fade', [
            state('visible', style({ opacity: 1, height: '*', width: '*' })),
            state('hidden', style({ opacity: 0, overflow: 'hidden', height: 0, width: 0 })),
            transition('hidden => visible', [
                animate('250ms ease-in', style({ opacity: 1, height: '*', width: '*' }))
            ]),
            transition('visible => hidden', [
                animate('350ms ease-out', style({ opacity: 0, width: 0, height: 0 }))
            ])
        ]),
        trigger('opacity', [
            transition('hidden => visible', [
                animate('450ms ease-in', keyframes([
                    style({ opacity: 0, offset: 0 }),
                    style({ opacity: 1, offset: 1 })
                ]))
            ]),
            transition('visible => hidden', [
                animate('250ms ease-out', keyframes([
                    style({ opacity: 1, offset: 0 }),
                    style({ opacity: 0.5, offset: 0.3 }),
                    style({ opacity: 0, offset: 1 })
                ]))
            ])
        ])
    ] });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "13.0.3", ngImport: i0, type: Ng2DropdownMenu, decorators: [{
            type: Component,
            args: [{ selector: 'ng2-dropdown-menu', template: `
        <!-- MENU -->
        <div
            class="ng2-dropdown-menu ng2-dropdown-menu---width--{{ width }}"
            [class.ng2-dropdown-menu--inside-element]="!appendToBody"
            [class.ng2-dropdown-menu--open]="dropdownState.menuState.isVisible"
            [style.z-index]="zIndex"
            [@fade]="dropdownState.menuState.toString()"
        >
            <div
                class="ng2-dropdown-menu__options-container"
                [@opacity]="dropdownState.menuState.toString()"
            >
                <ng-content></ng-content>
            </div>
        </div>

        <!-- BACKDROP -->
        <div
            class="ng2-dropdown-backdrop"
            *ngIf="dropdownState.menuState.isVisible"
            (click)="hide()"
        ></div>
    `, animations: [
                        trigger('fade', [
                            state('visible', style({ opacity: 1, height: '*', width: '*' })),
                            state('hidden', style({ opacity: 0, overflow: 'hidden', height: 0, width: 0 })),
                            transition('hidden => visible', [
                                animate('250ms ease-in', style({ opacity: 1, height: '*', width: '*' }))
                            ]),
                            transition('visible => hidden', [
                                animate('350ms ease-out', style({ opacity: 0, width: 0, height: 0 }))
                            ])
                        ]),
                        trigger('opacity', [
                            transition('hidden => visible', [
                                animate('450ms ease-in', keyframes([
                                    style({ opacity: 0, offset: 0 }),
                                    style({ opacity: 1, offset: 1 })
                                ]))
                            ]),
                            transition('visible => hidden', [
                                animate('250ms ease-out', keyframes([
                                    style({ opacity: 1, offset: 0 }),
                                    style({ opacity: 0.5, offset: 0.3 }),
                                    style({ opacity: 0, offset: 1 })
                                ]))
                            ])
                        ])
                    ], styles: [":host{display:block}.ng2-dropdown-menu{overflow-y:auto;box-shadow:0 1px 2px #0000004d;position:absolute;padding:.5em 0;background:#fff;border-radius:1px;max-height:400px;width:260px;min-height:0;display:block}.ng2-dropdown-menu.ng2-dropdown-menu--inside-element{position:fixed}.ng2-dropdown-menu.ng2-dropdown-menu--width--2{width:200px}.ng2-dropdown-menu.ng2-dropdown-menu--width--4{width:260px}.ng2-dropdown-menu.ng2-dropdown-menu--width--6{width:320px}.ng2-dropdown-backdrop{position:fixed;top:0;left:0;width:100%;height:100%;z-index:1;overflow:hidden}:host ::ng-deep .ng2-menu-divider{height:1px;min-height:1px;max-height:1px;width:100%;display:block;background:#f9f9f9}\n"] }]
        }], ctorParameters: function () { return [{ type: i1.DropdownStateService }, { type: i0.ElementRef }, { type: i0.Renderer2 }]; }, propDecorators: { width: [{
                type: Input
            }], focusFirstElement: [{
                type: Input
            }], offset: [{
                type: Input
            }], appendToBody: [{
                type: Input
            }], zIndex: [{
                type: Input
            }], items: [{
                type: ContentChildren,
                args: [Ng2MenuItem, { descendants: true }]
            }] } });
//# sourceMappingURL=data:application/json;base64,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