"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Migrator = void 0;
exports.normalizeVersion = normalizeVersion;
exports.parseMigrationsOptions = parseMigrationsOptions;
exports.executeMigrations = executeMigrations;
exports.migrate = migrate;
const chalk = require("chalk");
const child_process_1 = require("child_process");
const enquirer_1 = require("enquirer");
const path_1 = require("path");
const semver_1 = require("semver");
const node_url_1 = require("node:url");
const util_1 = require("util");
const tree_1 = require("../../generators/tree");
const fileutils_1 = require("../../utils/fileutils");
const logger_1 = require("../../utils/logger");
const git_utils_1 = require("../../utils/git-utils");
const package_json_1 = require("../../utils/package-json");
const package_manager_1 = require("../../utils/package-manager");
const handle_errors_1 = require("../../utils/handle-errors");
const connect_to_nx_cloud_1 = require("../connect/connect-to-nx-cloud");
const output_1 = require("../../utils/output");
const fs_1 = require("fs");
const workspace_root_1 = require("../../utils/workspace-root");
const is_ci_1 = require("../../utils/is-ci");
const installation_directory_1 = require("../../utils/installation-directory");
const configuration_1 = require("../../config/configuration");
const child_process_2 = require("../../utils/child-process");
const client_1 = require("../../daemon/client/client");
const nx_cloud_utils_1 = require("../../utils/nx-cloud-utils");
const project_graph_1 = require("../../project-graph/project-graph");
const format_changed_files_with_prettier_if_available_1 = require("../../generators/internal-utils/format-changed-files-with-prettier-if-available");
const execAsync = (0, util_1.promisify)(child_process_1.exec);
function normalizeVersion(version) {
    const [semver, ...prereleaseTagParts] = version.split('-');
    // Handle versions like 1.0.0-beta-next.2
    const prereleaseTag = prereleaseTagParts.join('-');
    const [major, minor, patch] = semver.split('.');
    const newSemver = `${major || 0}.${minor || 0}.${patch || 0}`;
    const newVersion = prereleaseTag
        ? `${newSemver}-${prereleaseTag}`
        : newSemver;
    const withoutPatch = `${major || 0}.${minor || 0}.0`;
    const withoutPatchAndMinor = `${major || 0}.0.0`;
    const variationsToCheck = [
        newVersion,
        newSemver,
        withoutPatch,
        withoutPatchAndMinor,
    ];
    for (const variation of variationsToCheck) {
        try {
            if ((0, semver_1.gt)(variation, '0.0.0')) {
                return variation;
            }
        }
        catch { }
    }
    return '0.0.0';
}
function cleanSemver(version) {
    return (0, semver_1.clean)(version) ?? (0, semver_1.coerce)(version);
}
function normalizeSlashes(packageName) {
    return packageName.replace(/\\/g, '/');
}
class Migrator {
    constructor(opts) {
        this.packageUpdates = {};
        this.collectedVersions = {};
        this.promptAnswers = {};
        this.packageJson = opts.packageJson;
        this.nxInstallation = opts.nxInstallation;
        this.getInstalledPackageVersion = opts.getInstalledPackageVersion;
        this.fetch = opts.fetch;
        this.installedPkgVersionOverrides = opts.from;
        this.to = opts.to;
        this.interactive = opts.interactive;
        this.excludeAppliedMigrations = opts.excludeAppliedMigrations;
    }
    async migrate(targetPackage, targetVersion) {
        await this.buildPackageJsonUpdates(targetPackage, {
            version: targetVersion,
            addToPackageJson: false,
        });
        const migrations = await this.createMigrateJson();
        return {
            packageUpdates: this.packageUpdates,
            migrations,
            minVersionWithSkippedUpdates: this.minVersionWithSkippedUpdates,
        };
    }
    async createMigrateJson() {
        const migrations = await Promise.all(Object.keys(this.packageUpdates).map(async (packageName) => {
            const currentVersion = this.getPkgVersion(packageName);
            if (currentVersion === null)
                return [];
            const { version } = this.packageUpdates[packageName];
            const { generators } = await this.fetch(packageName, version);
            if (!generators)
                return [];
            return Object.entries(generators)
                .filter(([, migration]) => migration.version &&
                this.gt(migration.version, currentVersion) &&
                this.lte(migration.version, version) &&
                this.areMigrationRequirementsMet(packageName, migration))
                .map(([migrationName, migration]) => ({
                ...migration,
                package: packageName,
                name: migrationName,
            }));
        }));
        return migrations.flat();
    }
    async buildPackageJsonUpdates(targetPackage, target) {
        const packagesToCheck = await this.populatePackageJsonUpdatesAndGetPackagesToCheck(targetPackage, target);
        for (const packageToCheck of packagesToCheck) {
            const filteredUpdates = {};
            for (const packageUpdate of packageToCheck.updates) {
                if (this.areRequirementsMet(packageUpdate.requires) &&
                    (!this.interactive ||
                        (await this.runPackageJsonUpdatesConfirmationPrompt(packageUpdate)))) {
                    Object.entries(packageUpdate.packages).forEach(([name, update]) => {
                        filteredUpdates[name] = update;
                        this.packageUpdates[name] = update;
                    });
                }
            }
            await Promise.all(Object.entries(filteredUpdates).map(([name, update]) => this.buildPackageJsonUpdates(name, update)));
        }
    }
    async populatePackageJsonUpdatesAndGetPackagesToCheck(targetPackage, target) {
        let targetVersion = target.version;
        if (this.to[targetPackage]) {
            targetVersion = this.to[targetPackage];
        }
        if (!this.getPkgVersion(targetPackage)) {
            this.addPackageUpdate(targetPackage, {
                version: target.version,
                addToPackageJson: target.addToPackageJson || false,
            });
            return [];
        }
        let migrationConfig;
        try {
            migrationConfig = await this.fetch(targetPackage, targetVersion);
        }
        catch (e) {
            if (e?.message?.includes('No matching version')) {
                throw new Error(`${e.message}\nRun migrate with --to="package1@version1,package2@version2"`);
            }
            else {
                throw e;
            }
        }
        targetVersion = migrationConfig.version;
        if (this.collectedVersions[targetPackage] &&
            (0, semver_1.gte)(this.collectedVersions[targetPackage], targetVersion)) {
            return [];
        }
        this.collectedVersions[targetPackage] = targetVersion;
        this.addPackageUpdate(targetPackage, {
            version: migrationConfig.version,
            addToPackageJson: target.addToPackageJson || false,
        });
        const { packageJsonUpdates, packageGroupOrder } = this.getPackageJsonUpdatesFromMigrationConfig(targetPackage, targetVersion, migrationConfig);
        if (!packageJsonUpdates.length) {
            return [];
        }
        const shouldCheckUpdates = packageJsonUpdates.some((packageJsonUpdate) => (this.interactive && packageJsonUpdate['x-prompt']) ||
            Object.keys(packageJsonUpdate.requires ?? {}).length);
        if (shouldCheckUpdates) {
            return [{ package: targetPackage, updates: packageJsonUpdates }];
        }
        const packageUpdatesToApply = packageJsonUpdates.reduce((m, c) => ({ ...m, ...c.packages }), {});
        return (await Promise.all(Object.entries(packageUpdatesToApply).map(([packageName, packageUpdate]) => this.populatePackageJsonUpdatesAndGetPackagesToCheck(packageName, packageUpdate))))
            .filter((pkgs) => pkgs.length)
            .flat()
            .sort((pkgUpdate1, pkgUpdate2) => packageGroupOrder.indexOf(pkgUpdate1.package) -
            packageGroupOrder.indexOf(pkgUpdate2.package));
    }
    getPackageJsonUpdatesFromMigrationConfig(packageName, targetVersion, migrationConfig) {
        const packageGroupOrder = this.getPackageJsonUpdatesFromPackageGroup(packageName, targetVersion, migrationConfig);
        if (!migrationConfig.packageJsonUpdates ||
            !this.getPkgVersion(packageName)) {
            return { packageJsonUpdates: [], packageGroupOrder };
        }
        const packageJsonUpdates = this.filterPackageJsonUpdates(migrationConfig.packageJsonUpdates, packageName, targetVersion);
        return { packageJsonUpdates, packageGroupOrder };
    }
    /**
     * Mutates migrationConfig, adding package group updates into packageJsonUpdates section
     *
     * @param packageName Package which is being migrated
     * @param targetVersion Version which is being migrated to
     * @param migrationConfig Configuration which is mutated to contain package json updates
     * @returns Order of package groups
     */
    getPackageJsonUpdatesFromPackageGroup(packageName, targetVersion, migrationConfig) {
        const packageGroup = packageName === '@nrwl/workspace' && (0, semver_1.lt)(targetVersion, '14.0.0-beta.0')
            ? LEGACY_NRWL_PACKAGE_GROUP
            : migrationConfig.packageGroup ?? [];
        let packageGroupOrder = [];
        if (packageGroup.length) {
            packageGroupOrder = packageGroup.map((packageConfig) => packageConfig.package);
            migrationConfig.packageJsonUpdates ??= {};
            const packages = {};
            migrationConfig.packageJsonUpdates[targetVersion + '--PackageGroup'] = {
                version: targetVersion,
                packages,
            };
            for (const packageConfig of packageGroup) {
                packages[packageConfig.package] = {
                    version: packageConfig.version === '*'
                        ? targetVersion
                        : packageConfig.version,
                    alwaysAddToPackageJson: false,
                };
                if (packageConfig.version === '*' &&
                    this.installedPkgVersionOverrides[packageName]) {
                    this.installedPkgVersionOverrides[packageConfig.package] ??=
                        this.installedPkgVersionOverrides[packageName];
                }
            }
        }
        return packageGroupOrder;
    }
    filterPackageJsonUpdates(packageJsonUpdates, packageName, targetVersion) {
        const filteredPackageJsonUpdates = [];
        for (const packageJsonUpdate of Object.values(packageJsonUpdates)) {
            if (!packageJsonUpdate.packages ||
                this.lt(packageJsonUpdate.version, this.getPkgVersion(packageName)) ||
                this.gt(packageJsonUpdate.version, targetVersion)) {
                continue;
            }
            const dependencies = {
                ...this.packageJson?.dependencies,
                ...this.packageJson?.devDependencies,
                ...this.nxInstallation?.plugins,
                ...(this.nxInstallation && { nx: this.nxInstallation.version }),
            };
            const filtered = {};
            for (const [packageName, packageUpdate] of Object.entries(packageJsonUpdate.packages)) {
                if (this.shouldApplyPackageUpdate(packageUpdate, packageName, dependencies)) {
                    filtered[packageName] = {
                        version: packageUpdate.version,
                        addToPackageJson: packageUpdate.alwaysAddToPackageJson
                            ? 'dependencies'
                            : packageUpdate.addToPackageJson || false,
                    };
                }
            }
            if (Object.keys(filtered).length) {
                packageJsonUpdate.packages = filtered;
                filteredPackageJsonUpdates.push(packageJsonUpdate);
            }
        }
        return filteredPackageJsonUpdates;
    }
    shouldApplyPackageUpdate(packageUpdate, packageName, dependencies) {
        return ((!packageUpdate.ifPackageInstalled ||
            this.getPkgVersion(packageUpdate.ifPackageInstalled)) &&
            (packageUpdate.alwaysAddToPackageJson ||
                packageUpdate.addToPackageJson ||
                !!dependencies?.[packageName]) &&
            (!this.collectedVersions[packageName] ||
                this.gt(packageUpdate.version, this.collectedVersions[packageName])));
    }
    addPackageUpdate(name, packageUpdate) {
        if (!this.packageUpdates[name] ||
            this.gt(packageUpdate.version, this.packageUpdates[name].version)) {
            this.packageUpdates[name] = packageUpdate;
        }
    }
    areRequirementsMet(requirements) {
        if (!requirements || !Object.keys(requirements).length) {
            return true;
        }
        return Object.entries(requirements).every(([pkgName, versionRange]) => {
            if (this.packageUpdates[pkgName]) {
                return (0, semver_1.satisfies)(cleanSemver(this.packageUpdates[pkgName].version), versionRange, { includePrerelease: true });
            }
            return (this.getPkgVersion(pkgName) &&
                (0, semver_1.satisfies)(this.getPkgVersion(pkgName), versionRange, {
                    includePrerelease: true,
                }));
        });
    }
    areMigrationRequirementsMet(packageName, migration) {
        if (!this.excludeAppliedMigrations) {
            return this.areRequirementsMet(migration.requires);
        }
        return ((this.wasMigrationSkipped(migration.requires) ||
            this.isMigrationForHigherVersionThanWhatIsInstalled(packageName, migration)) &&
            this.areRequirementsMet(migration.requires));
    }
    isMigrationForHigherVersionThanWhatIsInstalled(packageName, migration) {
        const installedVersion = this.getInstalledPackageVersion(packageName);
        return (migration.version &&
            (!installedVersion || this.gt(migration.version, installedVersion)) &&
            this.lte(migration.version, this.packageUpdates[packageName].version));
    }
    wasMigrationSkipped(requirements) {
        // no requiremets, so it ran before
        if (!requirements || !Object.keys(requirements).length) {
            return false;
        }
        // at least a requirement was not met, it was skipped
        return Object.entries(requirements).some(([pkgName, versionRange]) => !this.getInstalledPackageVersion(pkgName) ||
            !(0, semver_1.satisfies)(this.getInstalledPackageVersion(pkgName), versionRange, {
                includePrerelease: true,
            }));
    }
    async runPackageJsonUpdatesConfirmationPrompt(packageUpdate) {
        if (!packageUpdate['x-prompt']) {
            return Promise.resolve(true);
        }
        const promptKey = this.getPackageUpdatePromptKey(packageUpdate);
        if (this.promptAnswers[promptKey] !== undefined) {
            // a same prompt was already answered, skip
            return Promise.resolve(false);
        }
        return await (0, enquirer_1.prompt)([
            {
                name: 'shouldApply',
                type: 'confirm',
                message: packageUpdate['x-prompt'],
                initial: true,
            },
        ]).then(({ shouldApply }) => {
            this.promptAnswers[promptKey] = shouldApply;
            if (!shouldApply &&
                (!this.minVersionWithSkippedUpdates ||
                    (0, semver_1.lt)(packageUpdate.version, this.minVersionWithSkippedUpdates))) {
                this.minVersionWithSkippedUpdates = packageUpdate.version;
            }
            return shouldApply;
        });
    }
    getPackageUpdatePromptKey(packageUpdate) {
        return Object.entries(packageUpdate.packages)
            .map(([name, update]) => `${name}:${JSON.stringify(update)}`)
            .join('|');
    }
    getPkgVersion(pkg) {
        return this.getInstalledPackageVersion(pkg, this.installedPkgVersionOverrides);
    }
    gt(v1, v2) {
        return (0, semver_1.gt)(normalizeVersion(v1), normalizeVersion(v2));
    }
    lt(v1, v2) {
        return (0, semver_1.lt)(normalizeVersion(v1), normalizeVersion(v2));
    }
    lte(v1, v2) {
        return (0, semver_1.lte)(normalizeVersion(v1), normalizeVersion(v2));
    }
}
exports.Migrator = Migrator;
const LEGACY_NRWL_PACKAGE_GROUP = [
    { package: '@nrwl/workspace', version: '*' },
    { package: '@nrwl/angular', version: '*' },
    { package: '@nrwl/cypress', version: '*' },
    { package: '@nrwl/devkit', version: '*' },
    { package: '@nrwl/eslint-plugin-nx', version: '*' },
    { package: '@nrwl/express', version: '*' },
    { package: '@nrwl/jest', version: '*' },
    { package: '@nrwl/linter', version: '*' },
    { package: '@nrwl/nest', version: '*' },
    { package: '@nrwl/next', version: '*' },
    { package: '@nrwl/node', version: '*' },
    { package: '@nrwl/nx-plugin', version: '*' },
    { package: '@nrwl/react', version: '*' },
    { package: '@nrwl/storybook', version: '*' },
    { package: '@nrwl/web', version: '*' },
    { package: '@nrwl/js', version: '*' },
    { package: 'nx-cloud', version: 'latest' },
    { package: '@nrwl/react-native', version: '*' },
    { package: '@nrwl/detox', version: '*' },
    { package: '@nrwl/expo', version: '*' },
    { package: '@nrwl/tao', version: '*' },
];
async function normalizeVersionWithTagCheck(pkg, version) {
    // This doesn't seem like a valid version, lets check if its a tag on the registry.
    if (version && !(0, semver_1.parse)(version)) {
        try {
            return (0, package_manager_1.resolvePackageVersionUsingRegistry)(pkg, version);
        }
        catch {
            // fall through to old logic
        }
    }
    return normalizeVersion(version);
}
async function versionOverrides(overrides, param) {
    const res = {};
    const promises = overrides.split(',').map((p) => {
        const split = p.lastIndexOf('@');
        if (split === -1 || split === 0) {
            throw new Error(`Incorrect '${param}' section. Use --${param}="package@version"`);
        }
        const selectedPackage = p.substring(0, split).trim();
        const selectedVersion = p.substring(split + 1).trim();
        if (!selectedPackage || !selectedVersion) {
            throw new Error(`Incorrect '${param}' section. Use --${param}="package@version"`);
        }
        return normalizeVersionWithTagCheck(selectedPackage, selectedVersion).then((version) => {
            res[normalizeSlashes(selectedPackage)] = version;
        });
    });
    await Promise.all(promises);
    return res;
}
async function parseTargetPackageAndVersion(args) {
    if (!args) {
        throw new Error(`Provide the correct package name and version. E.g., my-package@9.0.0.`);
    }
    if (args.indexOf('@') > -1) {
        const i = args.lastIndexOf('@');
        if (i === 0) {
            const targetPackage = args.trim();
            const targetVersion = 'latest';
            return { targetPackage, targetVersion };
        }
        else {
            const targetPackage = args.substring(0, i);
            const maybeVersion = args.substring(i + 1);
            if (!targetPackage || !maybeVersion) {
                throw new Error(`Provide the correct package name and version. E.g., my-package@9.0.0.`);
            }
            const targetVersion = await normalizeVersionWithTagCheck(targetPackage, maybeVersion);
            return { targetPackage, targetVersion };
        }
    }
    else {
        if (args === 'latest' ||
            args === 'next' ||
            args === 'canary' ||
            (0, semver_1.valid)(args) ||
            args.match(/^\d+(?:\.\d+)?(?:\.\d+)?$/)) {
            // Passing `nx` here may seem wrong, but nx and @nrwl/workspace are synced in version.
            // We could duplicate the ternary below, but its not necessary since they are equivalent
            // on the registry
            const targetVersion = await normalizeVersionWithTagCheck('nx', args);
            const targetPackage = !['latest', 'next', 'canary'].includes(args) &&
                (0, semver_1.lt)(targetVersion, '14.0.0-beta.0')
                ? '@nrwl/workspace'
                : 'nx';
            return {
                targetPackage,
                targetVersion,
            };
        }
        else {
            return {
                targetPackage: args,
                targetVersion: 'latest',
            };
        }
    }
}
async function parseMigrationsOptions(options) {
    if (options.runMigrations === '') {
        options.runMigrations = 'migrations.json';
    }
    if (!options.runMigrations) {
        const [from, to] = await Promise.all([
            options.from
                ? versionOverrides(options.from, 'from')
                : Promise.resolve({}),
            options.to
                ? await versionOverrides(options.to, 'to')
                : Promise.resolve({}),
        ]);
        const { targetPackage, targetVersion } = await parseTargetPackageAndVersion(options['packageAndVersion']);
        return {
            type: 'generateMigrations',
            targetPackage: normalizeSlashes(targetPackage),
            targetVersion,
            from,
            to,
            interactive: options.interactive,
            excludeAppliedMigrations: options.excludeAppliedMigrations,
        };
    }
    else {
        return {
            type: 'runMigrations',
            runMigrations: options.runMigrations,
            ifExists: options.ifExists,
        };
    }
}
function createInstalledPackageVersionsResolver(root) {
    const cache = {};
    function getInstalledPackageVersion(packageName, overrides) {
        try {
            if (overrides?.[packageName]) {
                return overrides[packageName];
            }
            if (!cache[packageName]) {
                const { packageJson, path } = (0, package_json_1.readModulePackageJson)(packageName, (0, installation_directory_1.getNxRequirePaths)());
                // old workspaces would have the temp installation of nx in the cache,
                // so the resolved package is not the one we need
                if (!path.startsWith(workspace_root_1.workspaceRoot)) {
                    throw new Error('Resolved a package outside the workspace root.');
                }
                cache[packageName] = packageJson.version;
            }
            return cache[packageName];
        }
        catch {
            // Support migrating old workspaces without nx package
            if (packageName === 'nx') {
                cache[packageName] = getInstalledPackageVersion('@nrwl/workspace', overrides);
                return cache[packageName];
            }
            return null;
        }
    }
    return getInstalledPackageVersion;
}
// testing-fetch-start
function createFetcher() {
    const migrationsCache = {};
    const resolvedVersionCache = {};
    function fetchMigrations(packageName, packageVersion, setCache) {
        const cacheKey = packageName + '-' + packageVersion;
        return Promise.resolve(resolvedVersionCache[cacheKey])
            .then((cachedResolvedVersion) => {
            if (cachedResolvedVersion) {
                return cachedResolvedVersion;
            }
            resolvedVersionCache[cacheKey] = (0, package_manager_1.resolvePackageVersionUsingRegistry)(packageName, packageVersion);
            return resolvedVersionCache[cacheKey];
        })
            .then((resolvedVersion) => {
            if (resolvedVersion !== packageVersion &&
                migrationsCache[`${packageName}-${resolvedVersion}`]) {
                return migrationsCache[`${packageName}-${resolvedVersion}`];
            }
            setCache(packageName, resolvedVersion);
            return getPackageMigrationsUsingRegistry(packageName, resolvedVersion);
        })
            .catch(() => {
            logger_1.logger.info(`Fetching ${packageName}@${packageVersion}`);
            return getPackageMigrationsUsingInstall(packageName, packageVersion);
        });
    }
    return function nxMigrateFetcher(packageName, packageVersion) {
        if (migrationsCache[`${packageName}-${packageVersion}`]) {
            return migrationsCache[`${packageName}-${packageVersion}`];
        }
        let resolvedVersion = packageVersion;
        let migrations;
        function setCache(packageName, packageVersion) {
            migrationsCache[packageName + '-' + packageVersion] = migrations;
        }
        migrations = fetchMigrations(packageName, packageVersion, setCache).then((result) => {
            if (result.schematics) {
                result.generators = { ...result.schematics, ...result.generators };
                delete result.schematics;
            }
            resolvedVersion = result.version;
            return result;
        });
        setCache(packageName, packageVersion);
        return migrations;
    };
}
// testing-fetch-end
async function getPackageMigrationsUsingRegistry(packageName, packageVersion) {
    // check if there are migrations in the packages by looking at the
    // registry directly
    const migrationsConfig = await getPackageMigrationsConfigFromRegistry(packageName, packageVersion);
    if (!migrationsConfig) {
        return {
            name: packageName,
            version: packageVersion,
        };
    }
    if (!migrationsConfig.migrations) {
        return {
            name: packageName,
            version: packageVersion,
            packageGroup: migrationsConfig.packageGroup,
        };
    }
    logger_1.logger.info(`Fetching ${packageName}@${packageVersion}`);
    // try to obtain the migrations from the registry directly
    return await downloadPackageMigrationsFromRegistry(packageName, packageVersion, migrationsConfig);
}
async function getPackageMigrationsConfigFromRegistry(packageName, packageVersion) {
    const result = await (0, package_manager_1.packageRegistryView)(packageName, packageVersion, 'nx-migrations ng-update dist --json');
    if (!result) {
        return null;
    }
    const json = JSON.parse(result);
    if (!json['nx-migrations'] && !json['ng-update']) {
        const registry = new node_url_1.URL('dist' in json ? json.dist.tarball : json.tarball)
            .hostname;
        // Registries other than npmjs and the local registry may not support full metadata via npm view
        // so throw error so that fetcher falls back to getting config via install
        if (!['registry.npmjs.org', 'localhost', 'artifactory'].some((v) => registry.includes(v))) {
            throw new Error(`Getting migration config from registry is not supported from ${registry}`);
        }
    }
    return (0, package_json_1.readNxMigrateConfig)(json);
}
async function downloadPackageMigrationsFromRegistry(packageName, packageVersion, { migrations: migrationsFilePath, packageGroup, }) {
    const { dir, cleanup } = (0, package_manager_1.createTempNpmDirectory)();
    let result;
    try {
        const { tarballPath } = await (0, package_manager_1.packageRegistryPack)(dir, packageName, packageVersion);
        const migrations = await (0, fileutils_1.extractFileFromTarball)((0, path_1.join)(dir, tarballPath), (0, path_1.join)('package', migrationsFilePath), (0, path_1.join)(dir, migrationsFilePath)).then((path) => (0, fileutils_1.readJsonFile)(path));
        result = { ...migrations, packageGroup, version: packageVersion };
    }
    catch {
        throw new Error(`Failed to find migrations file "${migrationsFilePath}" in package "${packageName}@${packageVersion}".`);
    }
    finally {
        await cleanup();
    }
    return result;
}
async function getPackageMigrationsUsingInstall(packageName, packageVersion) {
    const { dir, cleanup } = (0, package_manager_1.createTempNpmDirectory)();
    let result;
    try {
        const pmc = (0, package_manager_1.getPackageManagerCommand)((0, package_manager_1.detectPackageManager)(dir), dir);
        await execAsync(`${pmc.add} ${packageName}@${packageVersion}`, {
            cwd: dir,
        });
        const { migrations: migrationsFilePath, packageGroup, packageJson, } = readPackageMigrationConfig(packageName, dir);
        let migrations = undefined;
        if (migrationsFilePath) {
            migrations = (0, fileutils_1.readJsonFile)(migrationsFilePath);
        }
        result = { ...migrations, packageGroup, version: packageJson.version };
    }
    catch (e) {
        logger_1.logger.warn(`Unable to fetch migrations for ${packageName}@${packageVersion}: ${e.message}`);
    }
    finally {
        await cleanup();
    }
    return result;
}
function readPackageMigrationConfig(packageName, dir) {
    const { path: packageJsonPath, packageJson: json } = (0, package_json_1.readModulePackageJson)(packageName, (0, installation_directory_1.getNxRequirePaths)(dir));
    const config = (0, package_json_1.readNxMigrateConfig)(json);
    if (!config) {
        return { packageJson: json, migrations: null, packageGroup: [] };
    }
    try {
        const migrationFile = require.resolve(config.migrations, {
            paths: [(0, path_1.dirname)(packageJsonPath)],
        });
        return {
            packageJson: json,
            migrations: migrationFile,
            packageGroup: config.packageGroup,
        };
    }
    catch {
        return {
            packageJson: json,
            migrations: null,
            packageGroup: config.packageGroup,
        };
    }
}
async function createMigrationsFile(root, migrations) {
    await writeFormattedJsonFile((0, path_1.join)(root, 'migrations.json'), { migrations });
}
async function updatePackageJson(root, updatedPackages) {
    const packageJsonPath = (0, path_1.join)(root, 'package.json');
    if (!(0, fs_1.existsSync)(packageJsonPath)) {
        return;
    }
    const parseOptions = {};
    const json = (0, fileutils_1.readJsonFile)(packageJsonPath, parseOptions);
    Object.keys(updatedPackages).forEach((p) => {
        if (json.devDependencies?.[p]) {
            json.devDependencies[p] = updatedPackages[p].version;
            return;
        }
        if (json.dependencies?.[p]) {
            json.dependencies[p] = updatedPackages[p].version;
            return;
        }
        const dependencyType = updatedPackages[p].addToPackageJson;
        if (typeof dependencyType === 'string') {
            json[dependencyType] ??= {};
            json[dependencyType][p] = updatedPackages[p].version;
        }
    });
    await writeFormattedJsonFile(packageJsonPath, json, {
        appendNewLine: parseOptions.endsWithNewline,
    });
}
async function updateInstallationDetails(root, updatedPackages) {
    const nxJsonPath = (0, path_1.join)(root, 'nx.json');
    const parseOptions = {};
    const nxJson = (0, fileutils_1.readJsonFile)(nxJsonPath, parseOptions);
    if (!nxJson.installation) {
        return;
    }
    const nxVersion = updatedPackages.nx?.version;
    if (nxVersion) {
        nxJson.installation.version = nxVersion;
    }
    if (nxJson.installation.plugins) {
        for (const dep in nxJson.installation.plugins) {
            const update = updatedPackages[dep];
            if (update) {
                nxJson.installation.plugins[dep] = (0, semver_1.valid)(update.version)
                    ? update.version
                    : await (0, package_manager_1.resolvePackageVersionUsingRegistry)(dep, update.version);
            }
        }
    }
    await writeFormattedJsonFile(nxJsonPath, nxJson, {
        appendNewLine: parseOptions.endsWithNewline,
    });
}
async function isMigratingToNewMajor(from, to) {
    from = normalizeVersion(from);
    to = ['latest', 'next', 'canary'].includes(to) ? to : normalizeVersion(to);
    if (!(0, semver_1.valid)(from)) {
        from = await (0, package_manager_1.resolvePackageVersionUsingRegistry)('nx', from);
    }
    if (!(0, semver_1.valid)(to)) {
        to = await (0, package_manager_1.resolvePackageVersionUsingRegistry)('nx', to);
    }
    return (0, semver_1.major)(from) < (0, semver_1.major)(to);
}
function readNxVersion(packageJson) {
    return (packageJson?.devDependencies?.['nx'] ??
        packageJson?.dependencies?.['nx'] ??
        packageJson?.devDependencies?.['@nx/workspace'] ??
        packageJson?.dependencies?.['@nx/workspace'] ??
        packageJson?.devDependencies?.['@nrwl/workspace'] ??
        packageJson?.dependencies?.['@nrwl/workspace']);
}
async function generateMigrationsJsonAndUpdatePackageJson(root, opts) {
    const pmc = (0, package_manager_1.getPackageManagerCommand)();
    try {
        const rootPkgJsonPath = (0, path_1.join)(root, 'package.json');
        let originalPackageJson = (0, fs_1.existsSync)(rootPkgJsonPath)
            ? (0, fileutils_1.readJsonFile)(rootPkgJsonPath)
            : null;
        const originalNxJson = (0, configuration_1.readNxJson)();
        const from = originalNxJson.installation?.version ??
            readNxVersion(originalPackageJson);
        logger_1.logger.info(`Fetching meta data about packages.`);
        logger_1.logger.info(`It may take a few minutes.`);
        const migrator = new Migrator({
            packageJson: originalPackageJson,
            nxInstallation: originalNxJson.installation,
            getInstalledPackageVersion: createInstalledPackageVersionsResolver(root),
            fetch: createFetcher(),
            from: opts.from,
            to: opts.to,
            interactive: opts.interactive && !(0, is_ci_1.isCI)(),
            excludeAppliedMigrations: opts.excludeAppliedMigrations,
        });
        const { migrations, packageUpdates, minVersionWithSkippedUpdates } = await migrator.migrate(opts.targetPackage, opts.targetVersion);
        await updatePackageJson(root, packageUpdates);
        await updateInstallationDetails(root, packageUpdates);
        if (migrations.length > 0) {
            await createMigrationsFile(root, [
                ...addSplitConfigurationMigrationIfAvailable(from, packageUpdates),
                ...migrations,
            ]);
        }
        output_1.output.success({
            title: `The migrate command has run successfully.`,
            bodyLines: [
                `- package.json has been updated.`,
                migrations.length > 0
                    ? `- migrations.json has been generated.`
                    : `- There are no migrations to run, so migrations.json has not been created.`,
            ],
        });
        try {
            if (['nx', '@nrwl/workspace'].includes(opts.targetPackage) &&
                (await isMigratingToNewMajor(from, opts.targetVersion)) &&
                !(0, is_ci_1.isCI)() &&
                !(0, nx_cloud_utils_1.isNxCloudUsed)(originalNxJson)) {
                output_1.output.success({
                    title: 'Connect to Nx Cloud',
                    bodyLines: [
                        'Nx Cloud is a first-party CI companion for Nx projects. It improves critical aspects of CI:',
                        '- Speed: 30% - 70% faster CI',
                        '- Cost: 40% - 75% reduction in CI costs',
                        '- Reliability: by automatically identifying flaky tasks and re-running them',
                    ],
                });
                await (0, connect_to_nx_cloud_1.connectToNxCloudWithPrompt)('migrate');
                originalPackageJson = (0, fileutils_1.readJsonFile)((0, path_1.join)(root, 'package.json'));
            }
        }
        catch {
            // The above code is to remind folks when updating to a new major and not currently using Nx cloud.
            // If for some reason it fails, it shouldn't affect the overall migration process
        }
        output_1.output.log({
            title: 'Next steps:',
            bodyLines: [
                `- Make sure package.json changes make sense and then run '${pmc.install}',`,
                ...(migrations.length > 0
                    ? [`- Run '${pmc.exec} nx migrate --run-migrations'`]
                    : []),
                ...(opts.interactive && minVersionWithSkippedUpdates
                    ? [
                        `- You opted out of some migrations for now. Write the following command down somewhere to apply these migrations later:`,
                        `  nx migrate ${opts.targetVersion} --from ${opts.targetPackage}@${minVersionWithSkippedUpdates} --exclude-applied-migrations`,
                        `- To learn more go to https://nx.dev/recipes/tips-n-tricks/advanced-update`,
                    ]
                    : [
                        `- To learn more go to https://nx.dev/features/automate-updating-dependencies`,
                    ]),
                ...(showConnectToCloudMessage()
                    ? [
                        `- You may run '${pmc.run('nx', 'connect-to-nx-cloud')}' to get faster builds, GitHub integration, and more. Check out https://nx.app`,
                    ]
                    : []),
            ],
        });
    }
    catch (e) {
        output_1.output.error({
            title: `The migrate command failed.`,
        });
        throw e;
    }
}
async function writeFormattedJsonFile(filePath, content, options) {
    const formattedContent = await (0, format_changed_files_with_prettier_if_available_1.formatFilesWithPrettierIfAvailable)([{ path: filePath, content: JSON.stringify(content) }], workspace_root_1.workspaceRoot, { silent: true });
    if (formattedContent.has(filePath)) {
        (0, fs_1.writeFileSync)(filePath, formattedContent.get(filePath), {
            encoding: 'utf-8',
        });
    }
    else {
        (0, fileutils_1.writeJsonFile)(filePath, content, options);
    }
}
function addSplitConfigurationMigrationIfAvailable(from, packageJson) {
    if (!packageJson['@nrwl/workspace'])
        return [];
    if ((0, semver_1.gte)(packageJson['@nrwl/workspace'].version, '15.7.0-beta.0') &&
        (0, semver_1.lt)(normalizeVersion(from), '15.7.0-beta.0')) {
        return [
            {
                version: '15.7.0-beta.0',
                description: 'Split global configuration files into individual project.json files. This migration has been added automatically to the beginning of your migration set to retroactively make them work with the new version of Nx.',
                cli: 'nx',
                implementation: './src/migrations/update-15-7-0/split-configuration-into-project-json-files',
                package: '@nrwl/workspace',
                name: '15-7-0-split-configuration-into-project-json-files',
            },
        ];
    }
    else {
        return [];
    }
}
function showConnectToCloudMessage() {
    try {
        const nxJson = (0, configuration_1.readNxJson)();
        const defaultRunnerIsUsed = (0, connect_to_nx_cloud_1.onlyDefaultRunnerIsUsed)(nxJson);
        return !!defaultRunnerIsUsed;
    }
    catch {
        return false;
    }
}
function runInstall() {
    const pmCommands = (0, package_manager_1.getPackageManagerCommand)();
    // TODO: remove this
    if ((0, package_manager_1.detectPackageManager)() === 'npm') {
        process.env.npm_config_legacy_peer_deps ??= 'true';
    }
    output_1.output.log({
        title: `Running '${pmCommands.install}' to make sure necessary packages are installed`,
    });
    (0, child_process_1.execSync)(pmCommands.install, { stdio: [0, 1, 2], windowsHide: true });
}
async function executeMigrations(root, migrations, isVerbose, shouldCreateCommits, commitPrefix) {
    let initialDeps = getStringifiedPackageJsonDeps(root);
    const installDepsIfChanged = () => {
        const currentDeps = getStringifiedPackageJsonDeps(root);
        if (initialDeps !== currentDeps) {
            runInstall();
        }
        initialDeps = currentDeps;
    };
    const migrationsWithNoChanges = [];
    const sortedMigrations = migrations.sort((a, b) => {
        // special case for the split configuration migration to run first
        if (a.name === '15-7-0-split-configuration-into-project-json-files') {
            return -1;
        }
        if (b.name === '15-7-0-split-configuration-into-project-json-files') {
            return 1;
        }
        return (0, semver_1.lt)(normalizeVersion(a.version), normalizeVersion(b.version))
            ? -1
            : 1;
    });
    logger_1.logger.info(`Running the following migrations:`);
    sortedMigrations.forEach((m) => logger_1.logger.info(`- ${m.package}: ${m.name} (${m.description})`));
    logger_1.logger.info(`---------------------------------------------------------\n`);
    for (const m of sortedMigrations) {
        logger_1.logger.info(`Running migration ${m.package}: ${m.name}`);
        try {
            const { collection, collectionPath } = readMigrationCollection(m.package, root);
            if (!isAngularMigration(collection, collectionPath, m.name)) {
                const changes = await runNxMigration(root, collectionPath, collection, m.name);
                logger_1.logger.info(`Ran ${m.name} from ${m.package}`);
                logger_1.logger.info(`  ${m.description}\n`);
                if (changes.length < 1) {
                    logger_1.logger.info(`No changes were made\n`);
                    migrationsWithNoChanges.push(m);
                    continue;
                }
                logger_1.logger.info('Changes:');
                (0, tree_1.printChanges)(changes, '  ');
                logger_1.logger.info('');
            }
            else {
                const ngCliAdapter = await getNgCompatLayer();
                const { madeChanges, loggingQueue } = await ngCliAdapter.runMigration(root, m.package, m.name, (0, project_graph_1.readProjectsConfigurationFromProjectGraph)(await (0, project_graph_1.createProjectGraphAsync)()).projects, isVerbose);
                logger_1.logger.info(`Ran ${m.name} from ${m.package}`);
                logger_1.logger.info(`  ${m.description}\n`);
                if (!madeChanges) {
                    logger_1.logger.info(`No changes were made\n`);
                    migrationsWithNoChanges.push(m);
                    continue;
                }
                logger_1.logger.info('Changes:');
                loggingQueue.forEach((log) => logger_1.logger.info('  ' + log));
                logger_1.logger.info('');
            }
            if (shouldCreateCommits) {
                installDepsIfChanged();
                const commitMessage = `${commitPrefix}${m.name}`;
                try {
                    const committedSha = (0, git_utils_1.commitChanges)(commitMessage);
                    if (committedSha) {
                        logger_1.logger.info(chalk.dim(`- Commit created for changes: ${committedSha}`));
                    }
                    else {
                        logger_1.logger.info(chalk.red(`- A commit could not be created/retrieved for an unknown reason`));
                    }
                }
                catch (e) {
                    logger_1.logger.info(chalk.red(`- ${e.message}`));
                }
            }
            logger_1.logger.info(`---------------------------------------------------------`);
        }
        catch (e) {
            output_1.output.error({
                title: `Failed to run ${m.name} from ${m.package}. This workspace is NOT up to date!`,
            });
            throw e;
        }
    }
    if (!shouldCreateCommits) {
        installDepsIfChanged();
    }
    return migrationsWithNoChanges;
}
async function runMigrations(root, opts, args, isVerbose, shouldCreateCommits = false, commitPrefix) {
    if (!process.env.NX_MIGRATE_SKIP_INSTALL) {
        runInstall();
    }
    if (!__dirname.startsWith(workspace_root_1.workspaceRoot)) {
        // we are running from a temp installation with nx latest, switch to running
        // from local installation
        (0, child_process_2.runNxSync)(`migrate ${args.join(' ')}`, {
            stdio: ['inherit', 'inherit', 'inherit'],
            env: {
                ...process.env,
                NX_MIGRATE_SKIP_INSTALL: 'true',
                NX_MIGRATE_USE_LOCAL: 'true',
            },
        });
        return;
    }
    const migrationsExists = (0, fileutils_1.fileExists)(opts.runMigrations);
    if (opts.ifExists && !migrationsExists) {
        output_1.output.log({
            title: `Migrations file '${opts.runMigrations}' doesn't exist`,
        });
        return;
    }
    else if (!opts.ifExists && !migrationsExists) {
        throw new Error(`File '${opts.runMigrations}' doesn't exist, can't run migrations. Use flag --if-exists to run migrations only if the file exists`);
    }
    output_1.output.log({
        title: `Running migrations from '${opts.runMigrations}'` +
            (shouldCreateCommits ? ', with each applied in a dedicated commit' : ''),
    });
    const migrations = (0, fileutils_1.readJsonFile)((0, path_1.join)(root, opts.runMigrations)).migrations;
    const migrationsWithNoChanges = await executeMigrations(root, migrations, isVerbose, shouldCreateCommits, commitPrefix);
    if (migrationsWithNoChanges.length < migrations.length) {
        output_1.output.success({
            title: `Successfully finished running migrations from '${opts.runMigrations}'. This workspace is up to date!`,
        });
    }
    else {
        output_1.output.success({
            title: `No changes were made from running '${opts.runMigrations}'. This workspace is up to date!`,
        });
    }
}
function getStringifiedPackageJsonDeps(root) {
    try {
        const { dependencies, devDependencies } = (0, fileutils_1.readJsonFile)((0, path_1.join)(root, 'package.json'));
        return JSON.stringify([dependencies, devDependencies]);
    }
    catch {
        // We don't really care if the .nx/installation property changes,
        // whenever nxw is invoked it will handle the dep updates.
        return '';
    }
}
async function runNxMigration(root, collectionPath, collection, name) {
    const { path: implPath, fnSymbol } = getImplementationPath(collection, collectionPath, name);
    const fn = require(implPath)[fnSymbol];
    const host = new tree_1.FsTree(root, process.env.NX_VERBOSE_LOGGING === 'true', `migration ${collection.name}:${name}`);
    await fn(host, {});
    host.lock();
    const changes = host.listChanges();
    (0, tree_1.flushChanges)(root, changes);
    return changes;
}
async function migrate(root, args, rawArgs) {
    await client_1.daemonClient.stop();
    return (0, handle_errors_1.handleErrors)(process.env.NX_VERBOSE_LOGGING === 'true', async () => {
        const opts = await parseMigrationsOptions(args);
        if (opts.type === 'generateMigrations') {
            await generateMigrationsJsonAndUpdatePackageJson(root, opts);
        }
        else {
            await runMigrations(root, opts, rawArgs, args['verbose'], args['createCommits'], args['commitPrefix']);
        }
    });
}
function readMigrationCollection(packageName, root) {
    const collectionPath = readPackageMigrationConfig(packageName, root).migrations;
    const collection = (0, fileutils_1.readJsonFile)(collectionPath);
    collection.name ??= packageName;
    return {
        collection,
        collectionPath,
    };
}
function getImplementationPath(collection, collectionPath, name) {
    const g = collection.generators?.[name] || collection.schematics?.[name];
    if (!g) {
        throw new Error(`Unable to determine implementation path for "${collectionPath}:${name}"`);
    }
    const implRelativePathAndMaybeSymbol = g.implementation || g.factory;
    const [implRelativePath, fnSymbol = 'default'] = implRelativePathAndMaybeSymbol.split('#');
    let implPath;
    try {
        implPath = require.resolve(implRelativePath, {
            paths: [(0, path_1.dirname)(collectionPath)],
        });
    }
    catch (e) {
        // workaround for a bug in node 12
        implPath = require.resolve(`${(0, path_1.dirname)(collectionPath)}/${implRelativePath}`);
    }
    return { path: implPath, fnSymbol };
}
// TODO (v21): Remove CLI determination of Angular Migration
function isAngularMigration(collection, collectionPath, name) {
    const entry = collection.generators?.[name] || collection.schematics?.[name];
    const shouldBeNx = !!collection.generators?.[name];
    const shouldBeNg = !!collection.schematics?.[name];
    if (entry.cli && entry.cli !== 'nx' && collection.generators?.[name]) {
        output_1.output.warn({
            title: `The migration '${collection.name}:${name}' appears to be an Angular CLI migration, but is located in the 'generators' section of migrations.json.`,
            bodyLines: [
                'In Nx 21, migrations inside `generators` will be treated as Nx Devkit migrations and therefore may not run correctly if they are using Angular Devkit.',
                'If the migration should be run with Angular Devkit, please place the migration inside `schematics` instead.',
                "Please open an issue on the plugin's repository if you believe this is an error.",
            ],
        });
    }
    // Currently, if the cli property exists we listen to it. If its nx, its not an ng cli migration.
    // If the property is not set, we will fall back to our intuition.
    return entry.cli ? entry.cli !== 'nx' : !shouldBeNx && shouldBeNg;
}
const getNgCompatLayer = (() => {
    let _ngCliAdapter;
    return async function getNgCompatLayer() {
        if (!_ngCliAdapter) {
            _ngCliAdapter = await Promise.resolve().then(() => require('../../adapter/ngcli-adapter'));
            require('../../adapter/compat');
        }
        return _ngCliAdapter;
    };
})();
