"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.releaseCLIHandler = void 0;
exports.createAPI = createAPI;
const enquirer_1 = require("enquirer");
const node_fs_1 = require("node:fs");
const nx_json_1 = require("../../config/nx-json");
const file_map_utils_1 = require("../../project-graph/file-map-utils");
const project_graph_1 = require("../../project-graph/project-graph");
const output_1 = require("../../utils/output");
const handle_errors_1 = require("../../utils/handle-errors");
const changelog_1 = require("./changelog");
const config_1 = require("./config/config");
const deep_merge_json_1 = require("./config/deep-merge-json");
const filter_release_groups_1 = require("./config/filter-release-groups");
const version_plans_1 = require("./config/version-plans");
const publish_1 = require("./publish");
const git_1 = require("./utils/git");
const github_1 = require("./utils/github");
const print_config_1 = require("./utils/print-config");
const resolve_nx_json_error_message_1 = require("./utils/resolve-nx-json-error-message");
const shared_1 = require("./utils/shared");
const version_1 = require("./version");
const releaseCLIHandler = (args) => (0, handle_errors_1.handleErrors)(args.verbose, () => createAPI({})(args));
exports.releaseCLIHandler = releaseCLIHandler;
function createAPI(overrideReleaseConfig) {
    const releaseVersion = (0, version_1.createAPI)(overrideReleaseConfig);
    const releaseChangelog = (0, changelog_1.createAPI)(overrideReleaseConfig);
    const releasePublish = (0, publish_1.createAPI)(overrideReleaseConfig);
    return async function release(args) {
        const projectGraph = await (0, project_graph_1.createProjectGraphAsync)({ exitOnError: true });
        const nxJson = (0, nx_json_1.readNxJson)();
        const userProvidedReleaseConfig = (0, deep_merge_json_1.deepMergeJson)(nxJson.release ?? {}, overrideReleaseConfig ?? {});
        const hasVersionGitConfig = Object.keys(userProvidedReleaseConfig.version?.git ?? {}).length > 0;
        const hasChangelogGitConfig = Object.keys(userProvidedReleaseConfig.changelog?.git ?? {}).length > 0;
        if (hasVersionGitConfig || hasChangelogGitConfig) {
            const jsonConfigErrorPath = hasVersionGitConfig
                ? ['release', 'version', 'git']
                : ['release', 'changelog', 'git'];
            const nxJsonMessage = await (0, resolve_nx_json_error_message_1.resolveNxJsonConfigErrorMessage)(jsonConfigErrorPath);
            output_1.output.error({
                title: `The "release" top level command cannot be used with granular git configuration. Instead, configure git options in the "release.git" property in nx.json, or use the version, changelog, and publish subcommands or programmatic API directly.`,
                bodyLines: [nxJsonMessage],
            });
            process.exit(1);
        }
        // Apply default configuration to any optional user configuration
        const { error: configError, nxReleaseConfig } = await (0, config_1.createNxReleaseConfig)(projectGraph, await (0, file_map_utils_1.createProjectFileMapUsingProjectGraph)(projectGraph), userProvidedReleaseConfig);
        if (configError) {
            return await (0, config_1.handleNxReleaseConfigError)(configError);
        }
        // --print-config exits directly as it is not designed to be combined with any other programmatic operations
        if (args.printConfig) {
            return (0, print_config_1.printConfigAndExit)({
                userProvidedReleaseConfig,
                nxReleaseConfig,
                isDebug: args.printConfig === 'debug',
            });
        }
        const rawVersionPlans = await (0, version_plans_1.readRawVersionPlans)();
        if (args.specifier && rawVersionPlans.length > 0) {
            output_1.output.error({
                title: `A specifier option cannot be provided when using version plans.`,
                bodyLines: [
                    `To override this behavior, use the Nx Release programmatic API directly (https://nx.dev/features/manage-releases#using-the-programmatic-api-for-nx-release).`,
                ],
            });
            process.exit(1);
        }
        // These properties must never be undefined as this command should
        // always explicitly override the git operations of the subcommands.
        const shouldCommit = userProvidedReleaseConfig.git?.commit ?? true;
        const shouldStage = (shouldCommit || userProvidedReleaseConfig.git?.stageChanges) ?? false;
        const shouldTag = userProvidedReleaseConfig.git?.tag ?? true;
        const versionResult = await releaseVersion({
            ...args,
            stageChanges: shouldStage,
            gitCommit: false,
            gitTag: false,
            deleteVersionPlans: false,
        });
        const changelogResult = await releaseChangelog({
            ...args,
            versionData: versionResult.projectsVersionData,
            version: versionResult.workspaceVersion,
            stageChanges: shouldStage,
            gitCommit: false,
            gitTag: false,
            createRelease: false,
            deleteVersionPlans: false,
        });
        const { error: filterError, releaseGroups, releaseGroupToFilteredProjects, } = (0, filter_release_groups_1.filterReleaseGroups)(projectGraph, nxReleaseConfig, args.projects, args.groups);
        if (filterError) {
            output_1.output.error(filterError);
            process.exit(1);
        }
        await (0, version_plans_1.setResolvedVersionPlansOnGroups)(rawVersionPlans, releaseGroups, Object.keys(projectGraph.nodes), args.verbose);
        const planFiles = new Set();
        releaseGroups.forEach((group) => {
            if (group.resolvedVersionPlans) {
                if (group.name === config_1.IMPLICIT_DEFAULT_RELEASE_GROUP) {
                    output_1.output.logSingleLine(`Removing version plan files`);
                }
                else {
                    output_1.output.logSingleLine(`Removing version plan files for group ${group.name}`);
                }
                group.resolvedVersionPlans.forEach((plan) => {
                    if (!args.dryRun) {
                        (0, node_fs_1.rmSync)(plan.absolutePath, { recursive: true, force: true });
                        if (args.verbose) {
                            console.log(`Removing ${plan.relativePath}`);
                        }
                    }
                    else {
                        if (args.verbose) {
                            console.log(`Would remove ${plan.relativePath}, but --dry-run was set`);
                        }
                    }
                    planFiles.add(plan.relativePath);
                });
            }
        });
        const deletedFiles = Array.from(planFiles);
        if (deletedFiles.length > 0) {
            await (0, git_1.gitAdd)({
                changedFiles: [],
                deletedFiles,
                dryRun: args.dryRun,
                verbose: args.verbose,
            });
        }
        if (shouldCommit) {
            output_1.output.logSingleLine(`Committing changes with git`);
            const commitMessage = nxReleaseConfig.git.commitMessage;
            const commitMessageValues = (0, shared_1.createCommitMessageValues)(releaseGroups, releaseGroupToFilteredProjects, versionResult.projectsVersionData, commitMessage);
            await (0, git_1.gitCommit)({
                messages: commitMessageValues,
                additionalArgs: nxReleaseConfig.git.commitArgs,
                dryRun: args.dryRun,
                verbose: args.verbose,
            });
        }
        if (shouldTag) {
            output_1.output.logSingleLine(`Tagging commit with git`);
            // Resolve any git tags as early as possible so that we can hard error in case of any duplicates before reaching the actual git command
            const gitTagValues = (0, shared_1.createGitTagValues)(releaseGroups, releaseGroupToFilteredProjects, versionResult.projectsVersionData);
            (0, shared_1.handleDuplicateGitTags)(gitTagValues);
            for (const tag of gitTagValues) {
                await (0, git_1.gitTag)({
                    tag,
                    message: nxReleaseConfig.git.tagMessage,
                    additionalArgs: nxReleaseConfig.git.tagArgs,
                    dryRun: args.dryRun,
                    verbose: args.verbose,
                });
            }
        }
        const shouldCreateWorkspaceRelease = (0, changelog_1.shouldCreateGitHubRelease)(nxReleaseConfig.changelog.workspaceChangelog);
        let hasPushedChanges = false;
        let latestCommit;
        if (shouldCreateWorkspaceRelease && changelogResult.workspaceChangelog) {
            output_1.output.logSingleLine(`Pushing to git remote`);
            // Before we can create/update the release we need to ensure the commit exists on the remote
            await (0, git_1.gitPush)({
                dryRun: args.dryRun,
                verbose: args.verbose,
            });
            hasPushedChanges = true;
            output_1.output.logSingleLine(`Creating GitHub Release`);
            latestCommit = await (0, git_1.getCommitHash)('HEAD');
            await (0, github_1.createOrUpdateGithubRelease)(nxReleaseConfig.changelog.workspaceChangelog
                ? nxReleaseConfig.changelog.workspaceChangelog.createRelease
                : false, changelogResult.workspaceChangelog.releaseVersion, changelogResult.workspaceChangelog.contents, latestCommit, { dryRun: args.dryRun });
        }
        for (const releaseGroup of releaseGroups) {
            const shouldCreateProjectReleases = (0, changelog_1.shouldCreateGitHubRelease)(releaseGroup.changelog);
            if (shouldCreateProjectReleases && changelogResult.projectChangelogs) {
                const projects = args.projects?.length
                    ? // If the user has passed a list of projects, we need to use the filtered list of projects within the release group
                        Array.from(releaseGroupToFilteredProjects.get(releaseGroup))
                    : // Otherwise, we use the full list of projects within the release group
                        releaseGroup.projects;
                const projectNodes = projects.map((name) => projectGraph.nodes[name]);
                for (const project of projectNodes) {
                    const changelog = changelogResult.projectChangelogs[project.name];
                    if (!changelog) {
                        continue;
                    }
                    if (!hasPushedChanges) {
                        output_1.output.logSingleLine(`Pushing to git remote`);
                        // Before we can create/update the release we need to ensure the commit exists on the remote
                        await (0, git_1.gitPush)({
                            dryRun: args.dryRun,
                            verbose: args.verbose,
                        });
                        hasPushedChanges = true;
                    }
                    output_1.output.logSingleLine(`Creating GitHub Release`);
                    if (!latestCommit) {
                        latestCommit = await (0, git_1.getCommitHash)('HEAD');
                    }
                    await (0, github_1.createOrUpdateGithubRelease)(releaseGroup.changelog
                        ? releaseGroup.changelog.createRelease
                        : false, changelog.releaseVersion, changelog.contents, latestCommit, { dryRun: args.dryRun });
                }
            }
        }
        let hasNewVersion = false;
        // null means that all projects are versioned together but there were no changes
        if (versionResult.workspaceVersion !== null) {
            hasNewVersion = Object.values(versionResult.projectsVersionData).some((version) => version.newVersion !== null);
        }
        let shouldPublish = !!args.yes && !args.skipPublish && hasNewVersion;
        const shouldPromptPublishing = !args.yes && !args.skipPublish && !args.dryRun && hasNewVersion;
        if (shouldPromptPublishing) {
            shouldPublish = await promptForPublish();
        }
        if (shouldPublish) {
            await releasePublish(args);
        }
        else {
            output_1.output.logSingleLine('Skipped publishing packages.');
        }
        return versionResult;
    };
}
async function promptForPublish() {
    try {
        const reply = await (0, enquirer_1.prompt)([
            {
                name: 'confirmation',
                message: 'Do you want to publish these versions?',
                type: 'confirm',
            },
        ]);
        return reply.confirmation;
    }
    catch (e) {
        // Handle the case where the user exits the prompt with ctrl+c
        return false;
    }
}
