"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TEN_MEGABYTES = exports.DeletedFileChange = exports.WholeFileChange = void 0;
exports.isWholeFileChange = isWholeFileChange;
exports.isDeletedFileChange = isDeletedFileChange;
exports.calculateFileChanges = calculateFileChanges;
exports.readWorkspaceConfig = readWorkspaceConfig;
exports.defaultFileRead = defaultFileRead;
exports.readPackageJson = readPackageJson;
const child_process_1 = require("child_process");
const fs_1 = require("fs");
const path_1 = require("path");
const configuration_1 = require("../config/configuration");
const workspace_root_1 = require("../utils/workspace-root");
const fileutils_1 = require("../utils/fileutils");
const json_diff_1 = require("../utils/json-diff");
const project_graph_1 = require("./project-graph");
const angular_json_1 = require("../adapter/angular-json");
const ignore_1 = require("../utils/ignore");
const project_configuration_utils_1 = require("./utils/project-configuration-utils");
const package_json_1 = require("../plugins/package-json");
const workspace_context_1 = require("../utils/workspace-context");
const project_json_1 = require("../plugins/project-json/build-nodes/project-json");
class WholeFileChange {
    constructor() {
        this.type = 'WholeFileChange';
    }
}
exports.WholeFileChange = WholeFileChange;
class DeletedFileChange {
    constructor() {
        this.type = 'WholeFileDeleted';
    }
}
exports.DeletedFileChange = DeletedFileChange;
function isWholeFileChange(change) {
    return change.type === 'WholeFileChange';
}
function isDeletedFileChange(change) {
    return change.type === 'WholeFileDeleted';
}
function calculateFileChanges(files, allWorkspaceFiles, nxArgs, readFileAtRevision = defaultReadFileAtRevision, ignore = (0, ignore_1.getIgnoreObject)()) {
    files = files.filter((f) => !ignore.ignores(f));
    return files.map((f) => {
        const ext = (0, path_1.extname)(f);
        const file = allWorkspaceFiles.find((fileData) => fileData.file == f);
        const hash = file?.hash;
        return {
            file: f,
            ext,
            hash,
            getChanges: () => {
                if (!(0, fs_1.existsSync)((0, path_1.join)(workspace_root_1.workspaceRoot, f))) {
                    return [new DeletedFileChange()];
                }
                if (!nxArgs) {
                    return [new WholeFileChange()];
                }
                if (nxArgs.files && nxArgs.files.includes(f)) {
                    return [new WholeFileChange()];
                }
                switch (ext) {
                    case '.json':
                        try {
                            const atBase = readFileAtRevision(f, nxArgs.base);
                            const atHead = readFileAtRevision(f, nxArgs.head);
                            return (0, json_diff_1.jsonDiff)(JSON.parse(atBase), JSON.parse(atHead));
                        }
                        catch (e) {
                            return [new WholeFileChange()];
                        }
                    default:
                        return [new WholeFileChange()];
                }
            },
        };
    });
}
exports.TEN_MEGABYTES = 1024 * 10000;
function defaultReadFileAtRevision(file, revision) {
    try {
        const fileFullPath = `${workspace_root_1.workspaceRoot}${path_1.sep}${file}`;
        const gitRepositoryPath = (0, child_process_1.execSync)('git rev-parse --show-toplevel')
            .toString()
            .trim();
        const filePathInGitRepository = (0, path_1.relative)(gitRepositoryPath, fileFullPath)
            .split(path_1.sep)
            .join('/');
        return !revision
            ? (0, fs_1.readFileSync)(file, 'utf-8')
            : (0, child_process_1.execSync)(`git show ${revision}:${filePathInGitRepository}`, {
                maxBuffer: exports.TEN_MEGABYTES,
                stdio: ['pipe', 'pipe', 'ignore'],
                windowsHide: true,
            })
                .toString()
                .trim();
    }
    catch {
        return '';
    }
}
/**
 * TODO(v20): Remove this function
 * @deprecated To get projects use {@link retrieveProjectConfigurations} instead. This will be removed in v20.
 */
function readWorkspaceConfig(opts) {
    let configuration = null;
    const root = opts.path || process.cwd();
    const nxJson = (0, configuration_1.readNxJson)(root);
    try {
        const projectGraph = (0, project_graph_1.readCachedProjectGraph)();
        configuration = {
            ...nxJson,
            ...(0, project_graph_1.readProjectsConfigurationFromProjectGraph)(projectGraph),
        };
    }
    catch {
        configuration = {
            version: 2,
            projects: getProjectsSync(root, nxJson),
        };
    }
    if (opts.format === 'angularCli') {
        return (0, angular_json_1.toOldFormat)(configuration);
    }
    else {
        return configuration;
    }
}
function defaultFileRead(filePath) {
    return (0, fs_1.readFileSync)((0, path_1.join)(workspace_root_1.workspaceRoot, filePath), 'utf-8');
}
function readPackageJson() {
    try {
        return (0, fileutils_1.readJsonFile)(`${workspace_root_1.workspaceRoot}/package.json`);
    }
    catch {
        return {}; // if package.json doesn't exist
    }
}
/**
 * TODO(v20): Remove this function.
 */
function getProjectsSync(root, nxJson) {
    /**
     * We can't update projects that come from plugins anyways, so we are going
     * to ignore them for now. Plugins should add their own add/create/update methods
     * if they would like to use devkit to update inferred projects.
     */
    const patterns = [
        '**/project.json',
        'project.json',
        ...(0, package_json_1.getGlobPatternsFromPackageManagerWorkspaces)(root, fileutils_1.readJsonFile),
    ];
    const projectFiles = (0, workspace_context_1.globWithWorkspaceContextSync)(root, patterns);
    const rootMap = {};
    for (const projectFile of projectFiles) {
        if ((0, path_1.basename)(projectFile) === 'project.json') {
            const json = (0, fileutils_1.readJsonFile)(projectFile);
            const config = (0, project_json_1.buildProjectFromProjectJson)(json, projectFile);
            (0, project_configuration_utils_1.mergeProjectConfigurationIntoRootMap)(rootMap, config, undefined, undefined, true);
        }
        else if ((0, path_1.basename)(projectFile) === 'package.json') {
            const packageJson = (0, fileutils_1.readJsonFile)(projectFile);
            const config = (0, package_json_1.buildProjectConfigurationFromPackageJson)(packageJson, root, projectFile, nxJson);
            if (!rootMap[config.root]) {
                (0, project_configuration_utils_1.mergeProjectConfigurationIntoRootMap)(rootMap, 
                // Inferred targets, tags, etc don't show up when running generators
                // This is to help avoid running into issues when trying to update the workspace
                {
                    name: config.name,
                    root: config.root,
                }, undefined, undefined, true);
            }
        }
    }
    return (0, project_configuration_utils_1.readProjectConfigurationsFromRootMap)(rootMap);
}
