"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handleErrors = handleErrors;
const client_1 = require("../daemon/client/client");
const logger_1 = require("./logger");
const output_1 = require("./output");
async function handleErrors(isVerbose, fn) {
    try {
        const result = await fn();
        if (typeof result === 'number') {
            return result;
        }
        return 0;
    }
    catch (err) {
        err ||= new Error('Unknown error caught');
        if (err.constructor.name === 'UnsuccessfulWorkflowExecution') {
            logger_1.logger.error('The generator workflow failed. See above.');
        }
        else if (err.name === 'ProjectGraphError') {
            const projectGraphError = err;
            let title = projectGraphError.message;
            if (projectGraphError.cause &&
                typeof projectGraphError.cause === 'object' &&
                'message' in projectGraphError.cause) {
                title += ' ' + projectGraphError.cause.message + '.';
            }
            if (isVerbose) {
                title += ' See errors below.';
            }
            const bodyLines = isVerbose
                ? formatErrorStackAndCause(projectGraphError)
                : ['Pass --verbose to see the stacktraces.'];
            output_1.output.error({
                title,
                bodyLines: bodyLines,
            });
        }
        else {
            const lines = (err.message ? err.message : err.toString()).split('\n');
            const bodyLines = lines.slice(1);
            if (isVerbose) {
                bodyLines.push(...formatErrorStackAndCause(err));
            }
            else if (err.stack) {
                bodyLines.push('Pass --verbose to see the stacktrace.');
            }
            output_1.output.error({
                title: lines[0],
                bodyLines,
            });
        }
        if (client_1.daemonClient.enabled()) {
            client_1.daemonClient.reset();
        }
        return 1;
    }
}
function formatErrorStackAndCause(error) {
    return [
        error.stack || error.message,
        ...(error.cause && typeof error.cause === 'object'
            ? [
                'Caused by:',
                'stack' in error.cause
                    ? error.cause.stack.toString()
                    : error.cause.toString(),
            ]
            : []),
    ];
}
